using UnityEngine;
using System;
using TMPro;
using System.Collections;

public class Manager_LevelEditor : MonoBehaviour
{
    [SerializeField] bool isEditorMode = true;
    // TODO some of these should be in a UI script
    [SerializeField] GameObject[] editorModeUI; // Should be separated into a UI script, but it's fine for now
    [SerializeField] GameObject[] playModeUI; // It's kinda an overkill to have to refactor this
    [SerializeField] GameObject preparationMenu;
    [SerializeField] GameObject aimingInteractor;
    [SerializeField] int currentLevel = 0;
    [SerializeField] int totalLevels = 0;
    private bool isVerifyMode = false;
    private bool isLevel = false;
    [SerializeField] GameObject modeSwitcher;
    [SerializeField] Manager_CurrentCourseIndex managerCurrentCourseIndex;
    [SerializeField] Manager_Level levelManager;
    [SerializeField] Manager_Game gameManager;
    [SerializeField] Manager_Score scoreManager;
    [SerializeField] private UI_LevelSelector_Editor uiLevelSelector;
    [SerializeField] private UI_LevelHole_Editor uiLevelHole;
    [SerializeField] private UI_LevelPar_Editor levelPar;
    [SerializeField] private GameObject verifiedIndicator;
    [SerializeField] private GameObject[] editorResultButtons;
    [SerializeField] private GameObject[] playResultButtons;
    [SerializeField] private Animator anim;
    [SerializeField] private TextMeshProUGUI currentLevelNumberText;
    public event Action OnTestingModeActivated;
    private void Awake()
    {
        isVerifyMode = PlayerPrefs.GetInt("isVerifyMode") == 1;
        isLevel = PlayerPrefs.GetInt("isDownloaded") == 1;
        Debug.Log("isVerifyMode: " + isVerifyMode);
        if (isVerifyMode)
        {
            isEditorMode = false;
            modeSwitcher.SetActive(false);
        }
        verifiedIndicator.SetActive(isVerifyMode && !isLevel);
        // Initialize the UI
        StartCoroutine(InitializeUI());
    }

    private void UpdateResultButtons()
    {
        foreach (GameObject button in editorResultButtons)
        {
            button.SetActive(isEditorMode);
        }
        foreach (GameObject button in playResultButtons)
        {
            button.SetActive(!isEditorMode);
        }
    }

    private IEnumerator InitializeUI()
    {
        yield return new WaitForEndOfFrame(); // Wait for the end of the frame to ensure all UI elements are initialized

        UpdateUI();
        UpdateResultButtons();
    }

    // Gets if the game is in verify mode
    public bool IsVerifyMode()
    {
        return isVerifyMode;
    }

    // Set the verification state, it should be called when ending a map and thus, verified.
    // Or when making changes and thus, unverified as the player didn't complete the map.
    public void SetVerifyMode(bool isVerified)
    {
        if (isLevel) return; // Don't verify downloaded levels
        managerCurrentCourseIndex.UpdateVerifiedStatus(isVerified);
    }

    // Update the par value of the level
    public void UpdateParValue(int par)
    {
        uiLevelSelector.SetParValue(par);
    }

    // Update the level
    public void UpdateLevel(int level)
    {
        uiLevelSelector.SetLevel(level);
    }

    // Update the par value and level of the par value editor window
    public void UpdateParValueAndLevel()
    {
        int level = levelPar.GetHoleNumber();
        int par = levelPar.GetParValue();

        Debug.Log("Level: " + level + " Par: " + par);

        uiLevelSelector.SetParValue(par);
        uiLevelSelector.SetLevel(level);
    }

    // Update the Par value of the level
    public void UpdateLevelPar()
    {
        int level = levelPar.GetHoleNumber();
        int par = levelPar.GetParValue();
        int totalLevels = levelManager.GetTotalLevels();

        Debug.Log("Level: " + level + " Par: " + par);
        
        if (level <= totalLevels)
        {
            Manager_Level.Level levelData = levelManager.GetLevels()[level - 1];
            levelData.parValue = par;
            managerCurrentCourseIndex.UpdateHolesPar(levelData.levelNum, par, levelData.spawnPoint);
            currentLevel = levelData.levelNum - 1;
            UpdateLevel();
        }
    }
    // Adds a level
    public void AddLevel(Manager_Level.Level level, GameObject levelObject)
    {
        int par = level.parValue;
        int levelNum = level.levelNum;
        Transform spawnPoint = level.spawnPoint;

        HoleInfo hole = managerCurrentCourseIndex.AddHolePar(levelNum, par, spawnPoint);
        
        // Update the level object's position
        levelObject.transform.position = hole.GetPosition();
        // Assign the Transform to the level's spawnPoint
        level.spawnPoint = levelObject.transform;
        level.parValue = hole.parValue;

        totalLevels++;
        uiLevelHole.UpdateEditorButtonLevelState();
    }

    // Removes a level
    public void RemoveLevel(int level)
    {
        managerCurrentCourseIndex.RemoveHolesPar(level);
        for (int i = level + 1; i <= totalLevels; i++)
        {
            managerCurrentCourseIndex.ChangeHolesParKey(i, i - 1);
        }
        totalLevels--;
        uiLevelHole.UpdateEditorButtonLevelState();
    }

    // Updates the level
    public void UpdateLevel(int level, int par, Transform spawnPoint)
    {
        managerCurrentCourseIndex.UpdateHolesPar(level, par, spawnPoint);
    }

    // Get the curent level
    public int GetTotalLevels()
    {
        return totalLevels;
    }

    // Sets the current level with clamping
    public void SetLevel(int levelIndex)
    {
        currentLevel = Math.Clamp(levelIndex, 0, totalLevels - 1);
    }

    // Updates the level
    public void UpdateLevel()
    {
        uiLevelSelector.SetLevel(currentLevel);
        int levelPar = levelManager.GetLevelPar(currentLevel);
        uiLevelSelector.SetParValue(levelPar);
        LoadLevel();
        ShowLevelIndicator();
    }

    // Updates the level indicator
    public void SelectLevel()
    {
        currentLevel = uiLevelHole.GetHoleNumber() - 1;
        UpdateLevel();
    }

    public void ShowLevelIndicator()
    {
        if (!isEditorMode) return;
        currentLevelNumberText.text = "Hole " + (levelManager.GetCurrentLevel() + 1).ToString();
        anim.SetTrigger("Show");
    }

    // Goes to the next level using SetLevel
    public void NextLevel()
    {
        SetLevel(++currentLevel);
        UpdateLevel();
    }

    // Goes to the previous level using SetLevel
    public void PreviousLevel()
    {
        SetLevel(--currentLevel);
        UpdateLevel();
    }

    // Returns the current level
    public int GetCurrentLevel()
    {
        return currentLevel;
    }

    public void LoadLevel()
    {
        gameManager.SetAndLoadLevel(currentLevel);
    }

    // Activates or deactivates the UI elements based on the mode
    private void SetModeUI(bool isEditorMode, GameObject[] modeUI)
    {
        foreach (GameObject ui in modeUI)
        {
            ui.SetActive(isEditorMode);
        }
    }

    // Get the current mode
    public bool IsEditorMode()
    {
        return isEditorMode;
    }

    // Toggles between editor and play mode
    public void ToggleMode()
    {
        OnTestingModeActivated?.Invoke();
        isEditorMode = !isEditorMode;
        PlayerPrefs.SetInt("isPaused", 0); // Ensure the game is not paused
        UpdateUI();
    }

    private void UpdateUI()
    {
        if (isEditorMode)
        {
            Debug.Log("Switched to Editor Mode");
            SetModeUI(true, editorModeUI);
            SetModeUI(false, playModeUI);
            // Using the level setter, update the current pos to the level pos
            ObjectFollower camManager = Camera.main.GetComponent<ObjectFollower>();
            camManager.SetFollow(false);
            aimingInteractor.SetActive(false);
            SetLevel(levelManager.GetCurrentLevel());
            UpdateLevel();
        }
        else
        {
            Debug.Log("Switched to Play Mode");
            SetModeUI(false, editorModeUI);
            preparationMenu.SetActive(true);
            aimingInteractor.SetActive(true);
        }
    }
}
