using UnityEngine;
using System.Collections;

public class Obstacle_Booster : MonoBehaviour
{
    private Manager_Game gameManager;
    private Manager_Player playerManager;
    [SerializeField] private float force = 20f;
    private bool canPropel = true;
    [SerializeField] private float cooldownDuration = 0.5f;
    [SerializeField] private KeyManager.BoosterType boosterType = KeyManager.BoosterType.Impulse_Right;
    [SerializeField] private Animator animator;
    private void Awake()
    {
        gameManager = (Manager_Game)Object.FindAnyObjectByType<Manager_Game>();
        playerManager = gameManager.GetPlayerManager();
    }

    void OnTriggerEnter2D(Collider2D collision)
    {
        // If the player collides with the object, trigger the object's function
        if (collision.gameObject.CompareTag("Player"))
        {
            StartCoroutine(PropelWithCooldown());
        }
    }

    private IEnumerator PropelWithCooldown()
    {
        canPropel = false;
        PropelPlayer();
        yield return new WaitForSeconds(cooldownDuration);
        canPropel = true;
    }

    // After detecting the player propel the player based on the booster type
    private void PropelPlayer()
    {
        Vector2 direction = Vector2.zero;
        // Default "Activate" or if true, a boolean for the alt animation
        bool isAlternative = Random.value < 0.5f ? false : true;

        switch (boosterType)
        {
            case KeyManager.BoosterType.Impulse_Right:
                direction = Vector2.right;
                break;
            case KeyManager.BoosterType.Impulse_Left:
                direction = Vector2.left;
                break;
            case KeyManager.BoosterType.Impulse_Up:
                direction = Vector2.up;
                break;
            case KeyManager.BoosterType.Impulse_Down:
                direction = Vector2.down;
                break;
            case KeyManager.BoosterType.Impulse_UpRight:
                direction = new Vector2(1, 1).normalized;
                break;
            case KeyManager.BoosterType.Impulse_UpLeft:
                direction = new Vector2(-1, 1).normalized;
                break;
            case KeyManager.BoosterType.Impulse_DownRight:
                direction = new Vector2(1, -1).normalized;
                break;
            case KeyManager.BoosterType.Impulse_DownLeft:
                direction = new Vector2(-1, -1).normalized;
                break;
            case KeyManager.BoosterType.Random:
                direction = new Vector2(Random.Range(-1f, 1f), Random.Range(-1f, 1f)).normalized;
                break;
            case KeyManager.BoosterType.Random_Axis:
                    int axisIndex = Random.Range(0, 4); // 0 = left, 1 = right, 2 = down, 3 = up
                    switch (axisIndex)
                    {
                        case 0: direction = Vector2.left; break;
                        case 1: direction = Vector2.right; break;
                        case 2: direction = Vector2.down; break;
                        case 3: direction = Vector2.up; break;
                    }
                    break;

            case KeyManager.BoosterType.Random_Diagonals:
                int diagIndex = Random.Range(0, 4); // 0 = up-right, 1 = up-left, 2 = down-right, 3 = down-left
                switch (diagIndex)
                {
                    case 0: direction = new Vector2(1, 1).normalized; break;
                    case 1: direction = new Vector2(-1, 1).normalized; break;
                    case 2: direction = new Vector2(1, -1).normalized; break;
                    case 3: direction = new Vector2(-1, -1).normalized; break;
                }
                break;

            case KeyManager.BoosterType.Random_X:
                direction = isAlternative ? Vector2.left : Vector2.right;
                break;

            case KeyManager.BoosterType.Random_Y:
                direction =isAlternative ? Vector2.down : Vector2.up;
                break;

            case KeyManager.BoosterType.Random_Diagonal_UpRight:
                direction = isAlternative ? new Vector2(1, 1).normalized : new Vector2(-1, -1).normalized;
                break;

            case KeyManager.BoosterType.Random_Diagonal_UpLeft:
                direction = isAlternative ? new Vector2(-1, 1).normalized : new Vector2(1, -1).normalized;
                break; 
            default:
                Debug.LogWarning("Invalid booster type.");
                return; // Exit the method if the booster type is invalid
        }

        animator.SetTrigger("Activate");
        if (HasParameter(animator, "Alternative"))
        {
            animator.SetBool("Alternative", isAlternative);
        }
        playerManager.PropelPlayer(direction, force); // Adjust the force as needed
    }
    // Check if the animator has a parameter with the given name
    bool HasParameter(Animator animator, string paramName)
    {
        foreach (AnimatorControllerParameter param in animator.parameters)
        {
            if (param.name == paramName)
                return true;
        }
        return false;
    }

}

