using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.EventSystems;
using System;

public class PanZoom : MonoBehaviour
{
    private Vector3 touchStart;
    private Camera cam;

    [SerializeField] private float defaultOrthoSize = 8f;
    [SerializeField] private float zoomMax = 14f;
    [SerializeField] private float zoomMin = 7f;
    public float increment = 0.01f;
    public float wheelIncrement = 5f;
    public bool isExploring = true;
    private bool dragging = false;

    public float smoothFactor = 2f; // Smoother movement on mobile

    public event Action<bool> OnCameraMoved;
    public event Action<bool> OnResetCalled;

    void Start()
    {
        cam = Camera.main;
        Application.targetFrameRate = 60; // Or higher, depending on device capability
    }

    // Set the exploring mode, which allows the player to explore the map
    public void SetExploring(bool exploring)
    {
        isExploring = exploring;
        if (!isExploring)
        {
            dragging = false;
        }
    }

    void Update()
    {
        if (PlayerPrefs.GetInt("isPaused") == 1) return;
        if (!isExploring) return;

        // Prevent interaction if touching a UI button
        if (IsPointerOverBlockingUI() && !dragging)
        {
            //Debug.Log("Pointer is over UI");
            dragging = false;
            return;
        }

#if UNITY_EDITOR || UNITY_STANDALONE
        HandleMouseInput();
#else
        HandleTouchInput();
#endif

        // Zoom with mouse scroll (PC only)
        float scroll = Input.GetAxis("Mouse ScrollWheel");
        if (scroll != 0)
        {
            Zoom(scroll * wheelIncrement);
        }
    }

    // Handle Mouse Input (PC)
    private void HandleMouseInput()
    {
        if (Input.GetMouseButtonDown(0))
        {
            touchStart = cam.ScreenToWorldPoint(Input.mousePosition);
            dragging = true;
        }

        if (Input.GetMouseButtonUp(0))
        {
            dragging = false;
        }

        if (dragging)
        {
            Vector3 direction = touchStart - cam.ScreenToWorldPoint(Input.mousePosition);
            cam.transform.position = Vector3.Lerp(cam.transform.position, cam.transform.position + direction, Time.deltaTime * smoothFactor);
            OnCameraMoved?.Invoke(true);
        }
    }

    // Handle Touch Input (Mobile)
    private void HandleTouchInput()
    {
        if (Input.touchCount == 1)
        {
            Touch touch = Input.GetTouch(0);
            if (touch.phase == TouchPhase.Began)
            {
                touchStart = cam.ScreenToWorldPoint(touch.position);
                dragging = true;
            }
            else if (touch.phase == TouchPhase.Moved && dragging)
            {
                Vector3 direction = touchStart - cam.ScreenToWorldPoint(touch.position);
                cam.transform.position = Vector3.Lerp(cam.transform.position, cam.transform.position + direction, Time.deltaTime * smoothFactor);
            }
            else if (touch.phase == TouchPhase.Ended || touch.phase == TouchPhase.Canceled)
            {
                dragging = false;
            }
            OnCameraMoved?.Invoke(true);
        }
        else if (Input.touchCount == 2) // Pinch-to-zoom
        {
            Touch touchZero = Input.GetTouch(0);
            Touch touchOne = Input.GetTouch(1);

            Vector2 touchZeroPrevPos = touchZero.position - touchZero.deltaPosition;
            Vector2 touchOnePrevPos = touchOne.position - touchOne.deltaPosition;

            float prevMagnitude = (touchZeroPrevPos - touchOnePrevPos).magnitude;
            float currentMagnitude = (touchZero.position - touchOne.position).magnitude;

            float difference = currentMagnitude - prevMagnitude;
            Zoom(difference * increment);
        }
    }

    // Zoom in or out
    private void Zoom(float increment)
    {
        OnCameraMoved?.Invoke(true);
        cam.orthographicSize = Mathf.Clamp(cam.orthographicSize - increment, zoomMin, zoomMax);
    }

    // Reset the camera to the default position
    public void ResetView()
    {
        cam.orthographicSize = defaultOrthoSize;
        transform.rotation = Quaternion.identity;
        OnResetCalled?.Invoke(true);
    }

    // Check if the pointer is over a UI element
    private bool IsPointerOverBlockingUI()
    {
       
        PointerEventData pointerData;

        if (Input.touchCount > 0)
        {
            Touch touch = Input.GetTouch(0);
            pointerData = new PointerEventData(EventSystem.current)
            {
                position = touch.position
            };
        }
        else
        {
            pointerData = new PointerEventData(EventSystem.current)
            {
                position = Input.mousePosition
            };
        }

        List<RaycastResult> results = new List<RaycastResult>();
        EventSystem.current.RaycastAll(pointerData, results);

        foreach (var result in results)
        {
            var gameObject = result.gameObject;
            // Check if the gameObject is active in the hierarchy
            if (!gameObject.activeInHierarchy)
                continue; // Skip inactive objects

            var graphic = gameObject.GetComponent<UnityEngine.UI.Graphic>();
            if (graphic != null && graphic.raycastTarget)
            {
                // Blocking UI detected
                return true;
            }
        }

        // No blocking UI found
        return false;
    }
}
