using UnityEngine;
using System.Collections;

public class Manager_Game : MonoBehaviour
{
  // References to the smaller components
  public Manager_Level levelManager;
  public Manager_Player playerManager;
  public Manager_Camera cameraManager;
  public Manager_Score scoreManager;
  public Manager_LevelEditor editorManager = null;
  private Coroutine delAppearCoroutine = null;
  private bool firstLoad = true;
  private bool editorMode = false;

  private void Awake()
  {
    // Get the editor manager if it exists
    if (editorManager != null)
    {
      editorMode = editorManager.IsEditorMode(); // User shouldn't be able to place things near the player or directly don't show the player
    }
    levelManager = FindObjectOfType<Manager_Level>();
  }

  public void Start()
  {
    StartCoroutine(Initialize());
  }

  private IEnumerator Initialize()
  {
    yield return new WaitForEndOfFrame();
    // Load the first level
    LoadLevel();
  }

  public void NextLevel()
  {
    int currentLevel = levelManager.GetCurrentLevel();

    // Update the score notification and set the hole score of the current level
    scoreManager.UpdateScoreNotification();
    scoreManager.SetHoleScore(currentLevel);

    // Go to the next level while stopping the player for repositioning
    levelManager.NextLevel();
    playerManager.StopPlayerAndDie();

    // Reposition the player and update the UI
    StartCoroutine(NextLevelCoroutine());
  }

  // Show the score of the current level and go to the next level
  private IEnumerator NextLevelCoroutine()
  {
    int newLevel = levelManager.GetCurrentLevel();
    int totalLevels = levelManager.GetTotalLevels();

    // Show the score notification
    scoreManager.ShowScoreAndUpdateHole(newLevel, totalLevels);

    yield return new WaitForSecondsRealtime(1.3f);
    LoadLevel();
    // Update score related data and UI
    scoreManager.ResetStrokes();
    if (newLevel < totalLevels)
    {
      scoreManager.UpdateParIndicator(); // Update the par indicator
      scoreManager.SetHoleNumber(newLevel + 1); // Update the hole number with offset
    }
  }

  private void LoadLevel()
  {
    int currentLevel = levelManager.GetCurrentLevel();
    int totalLevels = levelManager.GetTotalLevels();

    if (editorManager != null && currentLevel < totalLevels)
    {
      int currentLevelPar = levelManager.GetLevelPar();
      editorManager.UpdateParValue(currentLevelPar);
    }

    // If we passed all levels, stop the timer and end the game
    if (currentLevel >= totalLevels)
    {
      Debug.Log("Ending game...");
      scoreManager.StopTimer();
      StartCoroutine(StopGame());
      scoreManager.HideAll();
      return;
    }

    Debug.Log("Loading Level" + currentLevel);
    Transform newSpawnPoint = levelManager.GetLevelSpawnPoint();

    Debug.Log("repositioning" + newSpawnPoint.position.x + " " + newSpawnPoint.position.y);
    // Set the player to the new level spawn point and update the camera
    playerManager.LoadLevel(newSpawnPoint);
    cameraManager.ResetView(); // Reset the camera to the player position

    if (delAppearCoroutine != null)
    {
      StopCoroutine(delAppearCoroutine); // Stop any previous coroutine to avoid multiple calls
      delAppearCoroutine = null; // Clear the reference
    }
    delAppearCoroutine = StartCoroutine(DelayedPlayerAppear());
  }

  private IEnumerator DelayedPlayerAppear()
  {
    yield return new WaitForSeconds(0.5f); // Delay time in seconds
    playerManager.MakePlayerAppear(true, true);
      
    delAppearCoroutine = null; // Clear the reference to the coroutine
  }

  // Stop the game
  private IEnumerator StopGame()
  {
    yield return new WaitForSeconds(2f);
    EndGame();
  }

  // End the game
  public void EndGame()
  {
    Debug.Log("Game Ended!");

    // Hide the player to avoid any further movement
    GameObject player = GameObject.FindWithTag("Player");
    player.SetActive(false);

    // Save the score, and show the level summary by ending the game
    scoreManager.SaveScore(levelManager.GetLevelName());
    scoreManager.EndGame(levelManager.GetLevels());

    if (!editorMode)
    {
      if (editorManager != null && editorManager.IsVerifyMode())
      {
        Debug.Log("Level ended and verified!");
        editorManager.SetVerifyMode(true);
      }
    }
  }

  // Reset the level
  public void ResetLevel()
  {
    Debug.Log("Resetting level");
    playerManager.StopPlayerAndDie();
    StartCoroutine(ResetLevelCoroutine());
  }

  // Reset the level with a delay
  private IEnumerator ResetLevelCoroutine()
  {
    yield return new WaitForSeconds(0.8f);
    LoadLevel();
  }

  // Sets the current level and loads it
  public void SetAndLoadLevel(int levelNum)
  {
    levelManager.SetLevel(levelNum);
    LoadLevel();
  }

  // Adds a level, if it is the first, create it, otherwise, recreate it with the new level.
  public void AddLevel(Manager_Level.Level level)
  {
    // Add a level to the level manager
    levelManager.AddLevel(level);
  }

  // Removes a level by reference
  public void RemoveLevel(int levelNumToRemove)
  {
    if (levelManager == null || levelManager.GetLevels() == null || levelManager.GetLevels().Length == 0)
    {
      Debug.LogWarning("No levels to remove.");
      return;
    }
    levelManager.RemoveLevel(levelNumToRemove);
    LoadLevel();
    if (editorMode)
    {
      editorManager.UpdateLevel(levelManager.GetCurrentLevel());
      editorManager.UpdateParValue(levelManager.GetLevelPar());
    }
  }

  // Get the total levels
  public int GetTotalLevels()
  {
    return levelManager.GetTotalLevels();
  }

  public Manager_Player GetPlayerManager()
  {
    return playerManager;
  }
}