using Firebase.Auth;
using Firebase.Extensions;
using System;
using UnityEngine;

// Authenticates the user to firebase through login or register
public class CRUD_Authentication : MonoBehaviour
{
    private FirebaseAuth firebaseAuth = null;
    public event Action OnAuthSet; // Declare the event

    // Obtains the Firebase reference
    public void SetAuthRef(FirebaseAuth auth)
    {
        this.firebaseAuth = auth;
        Logout();
        // Trigger the event
        OnAuthSet?.Invoke();
    }

    // Tries to register the user with email, password, and nickname.
    public async void Register(string email, string password, string nickname, Action<FirebaseUser, string> onSuccess, Action<Exception> onError)
    {
        try
        {
            var result = await firebaseAuth.CreateUserWithEmailAndPasswordAsync(email, password);
            FirebaseUser newUser = result.User;
            UserProfile profile = new UserProfile { DisplayName = nickname };
            await newUser.UpdateUserProfileAsync(profile);
            onSuccess?.Invoke(newUser, nickname);

            if (newUser.IsEmailVerified)
            {
                Debug.Log("Email is verified! Proceed to the game.");
            }
            else
            {
                SendVerificationEmail(() => Debug.Log("Verification email sent"), (error) => Debug.LogError("Error sending verification email: " + error));
            }
        }
        catch (Exception ex)
        {
            onError?.Invoke(ex);
        }
    }

    // Sends a verification email to the user
    public async void SendVerificationEmail(Action onSuccess, Action<Exception> onError)
    {
        try
        {
            await firebaseAuth.CurrentUser.SendEmailVerificationAsync();
            onSuccess?.Invoke();
        }
        catch (Exception ex)
        {
            onError?.Invoke(ex);
        }
    }

    public void ResendVerificationEmail()
    {
        FirebaseUser user = FirebaseAuth.DefaultInstance.CurrentUser;
        if (user != null && !user.IsEmailVerified)
        {
            SendVerificationEmail(() => Debug.Log("Verification email sent"), (error) => Debug.LogError("Error sending verification email: " + error));
        }
        else
        {
            Debug.Log("User is already verified or not logged in.");
        }
    }

    // Tries to login the user with email and password.
    public async void Login(string email, string password, Action<FirebaseUser> onSuccess, Action<Exception> onError)
    {
        try
        {
            var result = await firebaseAuth.SignInWithEmailAndPasswordAsync(email, password);
            FirebaseUser newUser = result.User;
            onSuccess?.Invoke(newUser);
        }
        catch (Exception ex)
        {
            onError?.Invoke(ex);
        }
    }

    // Logs out the user
    public void Logout()
    {
        // Wait for the firbase to be initialized
        while (firebaseAuth == null)
        {
            Debug.Log("Waiting for firebase to be initialized...");
        }

        if (firebaseAuth.CurrentUser != null)
        {
            Debug.Log("Logging out user: " + firebaseAuth.CurrentUser.DisplayName);
            firebaseAuth.SignOut();
        }
    }

    // Checks if the user is logged in
    public bool IsUserLoggedIn()
    {
        return firebaseAuth != null && firebaseAuth.CurrentUser != null;
    }

    // Returns the user's display name
    public string GetUserDisplayName()
    {
        return firebaseAuth.CurrentUser.DisplayName;
    }

    // Returns the user's shortened ID
    public string GetUserID()
    {
        return firebaseAuth.CurrentUser.UserId;
    }
}
