using UnityEngine;
using Firebase.Auth;
using Firebase.Firestore;
using Firebase;
using System;

public class Manager_Auth : MonoBehaviour
{
    [SerializeField] private CRUD_Authentication auth;
    [SerializeField] private CRUD_Firestore firestoreInteractor;
    [SerializeField] private UI_Authentication uiAuth;
    [SerializeField] private Warning warning;
    [SerializeField] private string userID;
    public event Action OnLoggedIn; // Declare the event
    public event Action OnLoggedOut; // Declare the event

    public void Awake()
    {
        // Trigger the event
        auth.OnAuthSet += TryLogin;
    }
    public void TryLogin()
    {
        FirebaseUser currentUser = FirebaseAuth.DefaultInstance.CurrentUser;

        if (currentUser != null)
        {
            Debug.Log("User is logged in: " + currentUser.DisplayName);
            string displayName = currentUser.DisplayName; // The user's display name
            bool isEmailVerified = currentUser.IsEmailVerified; // Whether the email is verified
            if (!isEmailVerified)
            {
                return; // Stop login process
            }
            userID = currentUser.UserId; // The user's ID
            uiAuth.UpdatePanels(displayName, currentUser.UserId);
            OnLoggedIn?.Invoke(); // Trigger the event
        }
        else
        {
            auth.Logout();
            if (PlayerPrefs.HasKey("email") && PlayerPrefs.HasKey("password"))
            {
                AutoLogin(PlayerPrefs.GetString("email"), PlayerPrefs.GetString("password"));
            }
        }        
    }

    // Auto login the user
    public void AutoLogin(string email, string password)
    {
        Debug.Log("Auto login: " + email + " " + password);

        auth.Login(email, password, 
                (newUser) =>
                {
                    if (!newUser.IsEmailVerified)
                    {
                        Debug.LogWarning("Email not verified!");
                        uiAuth.DisplayLoginError("Please verify your email before logging in.", 0);
                        uiAuth.ShowVerificationPanel();
                        auth.ResendVerificationEmail();
                        return; // Stop login process
                    }

                    uiAuth.UpdatePanels(newUser.DisplayName, newUser.UserId);
                    userID = newUser.UserId;
                    uiAuth.CleanLoginErrors();
                    OnLoggedIn?.Invoke(); // Trigger the event
                    PlayerPrefs.SetInt("FailedLogin", 0);
                },
                (error) =>
                {
                    warning.ShowWarning("Failed to auto-login. Please login manually.", Warning.WarningType.Warning);
                    PlayerPrefs.SetInt("FailedLogin", 1);
                    return;
                });
    }

    // Register button clicked
    public void OnRegisterButtonClicked()
    {
        if (uiAuth.HasEmptyRegisterFields()) return;

        // Register to Firebase
        auth.Register(uiAuth.email.text, uiAuth.password.text, uiAuth.nickname.text, 
            (newUser, nickname) =>
            {
                if (!newUser.IsEmailVerified)
                {
                    Debug.LogWarning("Email not verified!");
                    uiAuth.ShowVerificationPanel();
                    auth.ResendVerificationEmail();
                    // Save the fields to avoid re-entering them
                    uiAuth.emailLogin.text = uiAuth.email.text;
                    uiAuth.passwordLogin.text = uiAuth.password.text;
                    return; // Stop login process
                }
                // TODO In the future save very basic info to Firestore
                uiAuth.UpdatePanels(newUser.DisplayName, newUser.UserId);
                uiAuth.CleanLoginErrors();
                userID = newUser.UserId;
                string email = uiAuth.email.text;
                string password = uiAuth.password.text;
                Debug.Log("Saving user: " + email + " " + password);
                SaveUser(email, password);
                PlayerPrefs.SetInt("FailedLogin", 0);
                CleanAfterFail();
                OnLoggedIn?.Invoke(); // Trigger the event

            },
            (error) => 
            {
                FirebaseException firebaseEx = error as FirebaseException;
                AuthError authError = (AuthError)firebaseEx.ErrorCode;
                int errorType = 0;

                string errorMessage = "Registration failed!";
                switch (authError)
                {
                    case AuthError.EmailAlreadyInUse:
                        errorMessage = "Email already in use!";
                        errorType = 0;
                        break;
                    case AuthError.InvalidEmail:
                        errorMessage = "Invalid email format!";
                        errorType = 0;
                        break;
                    case AuthError.WeakPassword:
                        errorMessage = "Password is too weak!";
                        errorType = 1;
                        break;
                    default:
                        errorMessage = "Registration failed! Check the fields and try again.";
                        errorType = 2;
                        break;
                }

                Debug.LogError(errorMessage);
                uiAuth.DisplayRegisterError(errorMessage, errorType);
            });
    }

    // Login button clicked
    public void OnLoginButtonClicked()
    {
        if (uiAuth.HasEmptyLoginFields()) return;

        // Login to Firebase
        auth.Login(uiAuth.emailLogin.text, uiAuth.passwordLogin.text,
            (newUser) =>
            {
                if (!newUser.IsEmailVerified)
                {
                    Debug.LogWarning("Email not verified!");
                    uiAuth.DisplayLoginError("Please verify your email before logging in.", 0);
                    uiAuth.ShowVerificationPanel();
                    auth.ResendVerificationEmail();
                    return; // Stop login process
                }

                uiAuth.UpdatePanels(newUser.DisplayName, newUser.UserId);
                uiAuth.CleanLoginErrors();
                warning.ShowWarning("Logged in successfully!", Warning.WarningType.Positive);
                userID = newUser.UserId;
                // Save the user's email and password
                string email = uiAuth.emailLogin.text;
                string password = uiAuth.passwordLogin.text;
                Debug.Log("Saving user on login: " + email + " " + password);

                SaveUser(email, password);
                CleanAfterFail();
                OnLoggedIn?.Invoke(); // Trigger the event
                PlayerPrefs.SetInt("FailedLogin", 0);
            },
            (error)  =>
            {
                FirebaseException firebaseEx = error as FirebaseException;
                AuthError authError = (AuthError)firebaseEx.ErrorCode;
                int errorType = 0;

                string errorMessage = "Login failed!";
                switch (authError)
                {
                    case AuthError.InvalidEmail:
                        errorMessage = "Invalid email format!";
                        errorType = 0;
                        break;
                    case AuthError.WrongPassword:
                        errorMessage = "Incorrect password!";
                        errorType = 1;
                        break;
                    case AuthError.UserNotFound:
                        errorMessage = "No user found!";
                        errorType = 0;
                        break;
                    case AuthError.UserDisabled:
                        errorMessage = "This account is disabled!";
                        errorType = 0;
                        break;
                    case AuthError.TooManyRequests:
                        errorMessage = "Too many failed login attempts. Try again later!";
                        errorType = 1;
                        break;
                    default:
                        errorMessage = "Login failed! Check the fields and try again.";
                        errorType = 1;
                        break;
                }

                Debug.LogError(errorMessage);
                uiAuth.DisplayLoginError(errorMessage, errorType);
            });
    }

    // Check if the use trying to login is verified
    public void CheckLoginVerification(GameObject verificationPanel)
    {
        auth.Login(uiAuth.emailLogin.text, uiAuth.passwordLogin.text,
        (newUser) =>
        {
            if (!newUser.IsEmailVerified)
            {
                Debug.LogWarning("Email not verified!");
                uiAuth.DisplayVerificationError("Email not verified!");
                return; // Stop login process
            }
            OnLoginButtonClicked();
            verificationPanel.SetActive(false);

        },
        (error) =>
        {
            Debug.LogError("Login failed: " + error.Message);
            uiAuth.DisplayVerificationError(error.Message);
        });

    }

    // Reset password button clicked
    public void ResetPassword()
    {
        string email = uiAuth.emailLogin.text;

        if (string.IsNullOrEmpty(email))
        {
            warning.ShowWarning("Please enter your email.", Warning.WarningType.Warning);
            return;
        }

        FirebaseAuth.DefaultInstance.SendPasswordResetEmailAsync(email).ContinueWith(task =>
        {
            if (task.IsCanceled || task.IsFaulted)
            {
                Debug.LogError("Failed to send password reset email.");
                UnityMainThreadDispatcher.Instance.Enqueue(() => { warning.ShowWarning("Failed to send password reset email.",  Warning.WarningType.Negative); });
            }
            else
            {
                Debug.Log("Password reset email sent!");
                UnityMainThreadDispatcher.Instance.Enqueue(() => { warning.ShowWarning("Password reset email sent. Please check your inbox.", Warning.WarningType.Warning); });
            }
        });
    }

    // Logout button clicked
    public void OnLogoutButtonClicked()
    {
        auth.Logout();
        uiAuth.CleanInputFields();
        uiAuth.SetAuthPanelsState(false);
        CleanSavedUser();
        warning.ShowWarning("Logged out successfully!", Warning.WarningType.Positive);
        userID = "";
        OnLoggedOut?.Invoke(); // Trigger the event
    }

    // If the user tries to login or register after a failed attempt, clean the data so it doesn't sync local data to the new account as the current data is already binded to the old account
    public void CleanAfterFail()
    {
        if (PlayerPrefs.GetInt("FailedLogin") == 1)
        {
            CleanUserData();
        }
    }

    // Clean user data
    public void CleanUserData()
    {
        auth.Logout();
        uiAuth.CleanInputFields();
        uiAuth.SetAuthPanelsState(false);
        CleanSavedUser();
        userID = "";
        OnLoggedOut?.Invoke();
    }

    // Check if user is logged in
    public bool IsUserLoggedIn()
    {
        return auth.IsUserLoggedIn();
    }

    // Try to login
    public void ShowLogin()
    {
        uiAuth.CleanInputFields();
        uiAuth.SetAuthPanelsState(false);
        uiAuth.ShowLoginPanel();
    }

    // Get the user's display name
    public string GetUserDisplayName()
    {
        return auth.GetUserDisplayName();
    }

    // Save the user's email and password
    private void SaveUser(string email, string password)
    {
        PlayerPrefs.SetString("email", email);
        PlayerPrefs.SetString("password", password);
    }

    // Clean the saved user's email and password
    private void CleanSavedUser()
    {
        PlayerPrefs.DeleteKey("email");
        PlayerPrefs.DeleteKey("password");
    }

    public string GetUserID()
    {
        return userID;
    }

}
