using UnityEngine;
using Firebase.Auth;
using Firebase.Firestore;
using System;
using System.Collections.Generic;
using System.Collections;

public class Manager_MapAccess : MonoBehaviour
{
    [SerializeField] private CRUD_Firestore firestoreCRUD;
    [SerializeField] private CRUD_Storage storageCRUD;
    [SerializeField] private CRUD_Authentication uiAuth;
    [SerializeField] private Interact_MapDataDisplayLocal mapInteractor;
    [SerializeField] private Manager_MapDataLocal mapManager;
    [SerializeField] private Manager_Auth authManager;
    [SerializeField] private Warning warning;
    [SerializeField] private GameObject verifyCourseIndicator; // Reference to the verified course indicator
    [SerializeField] private CRUD_EasySave3_Cooldown interactionCooldown;
    

    bool isUploading = false;

    /******* CRUD operation handlers of Map *******/
    
    // Tries to upload a map to Firestore -> C / U
    public void TryUploadMap()
    {
        if (!mapManager.IsMapVerified())
        {
            verifyCourseIndicator.SetActive(true);
            return;
        }

        if (!uiAuth.IsUserLoggedIn())
        {
            warning.ShowWarning("You need to be logged in to upload a map.");
            authManager.ShowLogin();
            return;
        }

        if (isUploading)
        {
            warning.ShowWarning("Please wait for the current upload to finish.");
            return;
        }

        string mapAuthor = uiAuth.GetUserDisplayName();
        string mapAuthorID = uiAuth.GetUserID();

        if (DataIntegrityChecker.IsMissingData(new[] { mapAuthor, mapAuthorID }))
        {
            warning.ShowWarning("Failed to get user data.", Warning.WarningType.Negative);
            return;
        }

        MapData_ES3 mapIndexData = mapManager.GetMapDetails();

        if (mapIndexData == null)
        {
            warning.ShowWarning("Failed to get map metadata.", Warning.WarningType.Negative);
            return;
        }

        string mapTilesData = mapManager.GetMapTilesData(mapIndexData.key);

        if (string.IsNullOrEmpty(mapTilesData))
        {
            warning.ShowWarning("Failed to get map tiles data.", Warning.WarningType.Negative);
            return;
        }

        string mapKey = mapIndexData.key;

        string cooldownTime = interactionCooldown.TryUpdateInteraction(mapKey, CRUD_EasySave3_Cooldown.MapInteractionType.Upload);
        if (cooldownTime != "-1s")
        {
            warning.ShowWarning($"You need to wait {cooldownTime} before you can do that again.");
            return;
        }

        Dictionary<string, object> data = null;        
        
        isUploading = true;
        warning.ShowWarning("Uploading map, hang tight!");
        storageCRUD.UploadMapData(mapKey, mapTilesData, (downloadUrl) => {
        if (!string.IsNullOrEmpty(downloadUrl)) 
        {
            Debug.Log("Received URL: " + downloadUrl);
            data = FirestoreKeys.GetMapData(mapIndexData, mapAuthor, mapAuthorID, downloadUrl);
            
            firestoreCRUD.UploadMapToFirestore(mapKey, data, (success) =>
            {                
                Debug.Log("UploadMapToFirestore callback received. Success: " + success);
                isUploading = false;
                // Enqueue UI updates to run on the main thread
                if (success)
                {
                    Debug.Log("Map uploaded to Firestore successfully.");
                    mapManager.UpdateUploadStatus(true);
                    mapManager.PopoulateInteractionMenu(mapManager.GetMapDetails());
                    warning.ShowWarning("Map uploaded successfully.", Warning.WarningType.Positive);
                }
                else
                {
                    Debug.LogError("Map upload to Firestore failed.");
                }
            });
        }
        else
        {
            Debug.LogError("Failed to get URL from Firebase Storage.");
            warning.ShowWarning(downloadUrl, Warning.WarningType.Negative);
            isUploading = false;
        }
    });
    }

    // Tries to download a map from Firestore -> R
    public void TryDownloadMap(string mapName, string mapKey, Action onSuccess, Action<Exception> onError)
    {
        storageCRUD.DownloadMap(mapName, mapKey, onSuccess, onError);
    }

    // Try update map data -> U
    public void IsMapUpToDate(string mapKey, string lastMod, Action<MapData_Cloud, bool> onResultUpdateMap)
    {
        firestoreCRUD.IsMapUpToDate(mapKey, lastMod, onResultUpdateMap);
    }

    // Tries to delete a map and updates the upload status -> D
    public void TryDeleteMap(string mapID, Action<bool> onDeleteResult)
    {
        if (mapInteractor.IsMapUploaded())
        {
            // First try to delete from storage, then from Firestore
            storageCRUD.DeleteMap(mapID, result =>
            {
                if (result)
                {                   
                    firestoreCRUD.DeleteMap(mapID, result =>
                    {
                        onDeleteResult?.Invoke(result);
                    });
                }
                else
                {
                    warning.ShowWarning("Failed to delete map", Warning.WarningType.Negative);
                    onDeleteResult?.Invoke(false);
                }
            });
            return;
        }

        onDeleteResult?.Invoke(true);
    }
}
