using System;
using System.Collections.Generic;
using UnityEngine;
using TMPro;
using UnityEngine.UI;
using System.Linq;
using System.Globalization;

public class Manager_MapQuery : MonoBehaviour
{
    [SerializeField] private CloudMapFetch cloudDownload;
    [SerializeField] private GameObject mapDisplayPrefab;
    [SerializeField] private Transform mapsViewport;
    [SerializeField] private GameObject quickSearch;
    [SerializeField] private GameObject quickSearchButton;
    [SerializeField] private TMP_Text warningText;
    [SerializeField] private TMP_InputField searchQueryField;
    [SerializeField] private Warning warning;
    [SerializeField] private TMP_Dropdown filterDropdown;
    private int contentLength = 0;
    public ScrollRect scrollRect;
    private string previousQuery = "";
    private bool isSearching = false;
    private bool processingNextBatch = false;

    // Search parameters
    public enum SearchParam
    {
        Name,
        MapID,
        AuthorID,
        MostRecent,
        MostDownloaded,
        MostLiked,
        BestRated

    }

    // Default search parameter
    private SearchParam previousSearchParam = SearchParam.Name;
    private SearchParam currentSearchParam = SearchParam.Name;

    public void Start()
    {
        scrollRect.onValueChanged.AddListener(OnScroll);
    }

    // Update the content size TODO maybe move this to a separate script
    private void OnScroll(Vector2 scrollPosition)
    {
        if (0 == mapsViewport.childCount)
        {
            return;
        }
        float scrollY = scrollRect.verticalNormalizedPosition;
        float loadThreshold = 0.9f; // Load more when user scrolls 90% to the end
        bool isAtBottom = scrollY < (1 - loadThreshold);

        if (isAtBottom && mapsViewport.childCount >= 5)
        {
            if (!ProcessNextBatch())
            {
                warning.ShowWarning("No more maps to process...", Warning.WarningType.Warning);
                Debug.Log("No more maps to process.");
            }
        }
    }
    
    // By default, search for all maps
    public void DefaultSearch()
    {
        if (cloudDownload.IsFirestoreConnected())
        {
            CleanMaps();
            FindMapsByName("");
        }
        else
        {
            Debug.LogError("Firestore is not connected.");
        }
    }

    // Clean the maps
    public void CleanMaps()
    {
        foreach (Transform child in mapsViewport)
        {
            Destroy(child.gameObject);
        }
        filterDropdown.interactable = false; // Disable filtering
        filterDropdown.value = 0; // Reset the dropdown to the first option
    }

    public void FilterElements()
    {
        if (mapsViewport.childCount <= 0)
        {
            Debug.Log("No maps to filter.");
            return;
        }

        if (filterDropdown.value == 0)
        {
            Debug.Log("No filter.");
            return;
        }

        // Step 1: Gather all map displays
        List<Object_MapDisplayCloud> mapDisplays = new List<Object_MapDisplayCloud>();
        foreach (Transform child in mapsViewport)
        {
            Object_MapDisplayCloud display = child.GetComponent<Object_MapDisplayCloud>();
            if (display != null)
            {
                mapDisplays.Add(display);
            }
        }
        int selectedOption = filterDropdown.value;

        // Step 2: Sort the list based on the selected option
        switch (selectedOption)
        {
            case 1: // Most recent
                var format = "dd-MM-yyyy HH:mm:ss";
                mapDisplays = mapDisplays.OrderBy(m =>
                    DateTime.ParseExact(m.GetMapData().lastModifiedDate, format, CultureInfo.InvariantCulture)
                ).ToList();
                break;

            case 2: // Most downloaded
                mapDisplays = mapDisplays.OrderBy(m =>
                    m.GetMapData().downloads
                ).ToList();
                break;

            case 3: // Most liked
                mapDisplays = mapDisplays.OrderBy(m =>
                    m.GetMapData().likes
                ).ToList();
                break;

            case 4: // Best rated (rating = totalRating / rates)
                mapDisplays = mapDisplays.OrderBy(m =>
                {
                    var data = m.GetMapData();
                    return (data.rates > 0) ? ((float)data.totalRating / data.rates) : 0f;
                }).ToList();
                break;

            default:
                return;
        }

        // Step 3: Reorder the UI
        for (int i = 0; i < mapDisplays.Count; i++)
        {
            mapDisplays[i].transform.SetSiblingIndex(i);
        }
    }

    // Check if the query has changed
    private bool ChangedQuery(string query)
    {
       return query != previousQuery;
    }

    // Search for maps
    public void SearchInputField(SearchParam newSearchParam, bool forceSearch = true, bool quickSearchOp = false)
    {
        if (isSearching) return;
        string query = searchQueryField.text;
        if (string.IsNullOrEmpty(query) && !quickSearchOp)
        {
            quickSearch.SetActive(true);
            CleanMaps();
            warning.ShowWarning("Oops! You need to enter a search term first.", Warning.WarningType.Warning);
            warningText.text = "";
            return;
        }
        else
        {
            quickSearch.SetActive(false);
            quickSearchButton.SetActive(false);
        }

        warningText.text = "Searching for: " + query;

        bool queryChanged = ChangedQuery(query);
        bool paramChanged = newSearchParam != previousSearchParam;

        if (!queryChanged && !paramChanged && !forceSearch) return;

        currentSearchParam = newSearchParam;
        CleanMaps();
        isSearching = true;

        Debug.Log("Search parameter: " + newSearchParam);

        switch (newSearchParam)
        {
            case SearchParam.MapID:
                FindMapsByID(query);
                break;
            case SearchParam.AuthorID:
                FindMapsByAuthorID(query);
                break;
            case SearchParam.MostRecent:
                FindMostRecentMaps();
                break;
            case SearchParam.MostDownloaded:
                FindMostDownloadedMaps();
                break;
            case SearchParam.MostLiked:
                FindMostLikedMaps();
                break;
            case SearchParam.BestRated:
                FindBestRatedMaps();
                break;
            case SearchParam.Name:
            default:
                FindMapsByName(query);
                break;
        }
        previousSearchParam = newSearchParam;
        previousQuery = query;
    }

    public void Search()
    {
        SearchInputField(currentSearchParam);
    }

    // Search by name
    public void SearchByName()
    {
        SearchInputField(SearchParam.Name);
    }

    // Search by map ID
    public void SearchByMapID()
    {
        SearchInputField(SearchParam.MapID);
    }

    // Search by author's ID
    public void SearchByAuthorID()
    {
        Debug.Log("Searching by Author ID");
        SearchInputField(SearchParam.AuthorID);
    }

    // Search by placeholder
    public void SearchByMostRecent()
    {
        SearchInputField(SearchParam.MostRecent, forceSearch: true, quickSearchOp: true);
    }

        // Search by placeholder
    public void SearchByMostDownloaded()
    {
        SearchInputField(SearchParam.MostDownloaded, forceSearch: true, quickSearchOp: true);
    }

    // Search by placeholder
    public void SearchByBestRated()
    {
        SearchInputField(SearchParam.BestRated, forceSearch: true, quickSearchOp: true);
    }

    // Search by placeholder
    public void SearchByMostLiked()
    {
        SearchInputField(SearchParam.MostLiked, forceSearch: true, quickSearchOp: true);
    }

    // Refresh the search
    public void RefreshSearch()
    {
        SearchInputField(previousSearchParam, true);
    }

    // Find maps by author's ID
    private void FindMapsByAuthorID(string authorID)
    {
        Debug.Log("Searching for maps by author ID: " + authorID);
        cloudDownload.GetMapsByAuthorID(authorID, OnSuccess, OnError);
    }

    // Find maps by map ID
    private void FindMapsByID(string mapID)
    {
        Debug.Log("Searching for map with ID: " + mapID);
        cloudDownload.GetMapByID(mapID, OnSuccess, OnError);
    }

    // Find maps by name
    private void FindMapsByName(string mapName)
    {
        Debug.Log("Searching for maps with name: " + mapName);
        cloudDownload.GetMapsByName(mapName, OnSuccess, OnError);
    }

    // Find the most recent maps
    private void FindMostRecentMaps()
    {
        Debug.Log("Searching for the most RECENT maps.");
        warningText.text = "Searching for most recent maps...";
        cloudDownload.GetMostRecentMaps(OnSuccess, OnError);
    }

    // Find the most downloaded maps
    private void FindMostDownloadedMaps()
    {
        Debug.Log("Searching for the most DOWNLOADED maps.");
        warningText.text = "Searching for most downloaded maps...";
        cloudDownload.GetMostDownloadedMaps(OnSuccess, OnError);
    }

    // Find the most rated maps
    private void FindBestRatedMaps()
    {
        Debug.Log("Searching for the best RATED maps.");
        warningText.text = "Searching for most rated maps...";
        cloudDownload.GetBestRatedMaps(OnSuccess, OnError);
    }

    // Find the Most liked maps
    private void FindMostLikedMaps()
    {
        Debug.Log("Searching for the most LIKED maps.");
        warningText.text = "Searching for most liked maps...";
        cloudDownload.GetMostLikedMaps(OnSuccess, OnError);
    }
    private string GetSearchParamString(SearchParam searchParam)
    {
        switch (searchParam)
        {
            case SearchParam.Name:
                return searchQueryField.text;
            case SearchParam.MapID:
                return  searchQueryField.text;
            case SearchParam.AuthorID:
                return  searchQueryField.text;
            case SearchParam.MostRecent:
                return "Most Recent";
            case SearchParam.MostDownloaded:
                return "Most Downloaded";
            case SearchParam.MostLiked:
                return "Most Liked";
            case SearchParam.BestRated:
                return "Best Rated";
            default:
                return "";
        }
    }
    private void OnSuccess(List<MapData_Cloud> maps)
    {
        Debug.Log("OnSuccess callback received.");
        Debug.Log("Maps found: " + maps.Count);
        Debug.Log("Is searching: " + isSearching);
        isSearching = false;
        string searchQuery = GetSearchParamString(currentSearchParam);
        // If there are no maps, return
        if (maps.Count <= 0 && mapsViewport.childCount <= 0)
        {
            warningText.text = "No results found for: " + searchQuery;
            quickSearch.SetActive(false);
            quickSearchButton.SetActive(true); // Show the quick search button to allow users to search again
            Debug.Log("There are no maps to display.");
            filterDropdown.interactable = false; // Disable filtering
            return;
        }
        // If there are maps enable filtering and set it to the first option
        filterDropdown.interactable = true;
        filterDropdown.RefreshShownValue();

        warningText.text = "";
        foreach (var map in maps)
        {
            Debug.Log($"Map: {map.mapName}, Author: {map.mapAuthor}({map.mapAuthorID}), Description: {map.mapDescription}");
            // Create the map display object
            Instantiate(mapDisplayPrefab, mapsViewport).GetComponent<Object_MapDisplayCloud>().SetMapData(map);
        }
    }

    public bool ProcessNextBatch()
    {
        if (processingNextBatch)
            return false; // Prevent multiple calls

        if (mapsViewport.childCount <= 0)
            return Fail("No maps to process.");

        processingNextBatch = true;
        const int batchSize = 5;

        Debug.Log("Processing next batch...");
        cloudDownload.ProcessNextBatch(OnSuccess, batchSize);
        FilterElements();
        processingNextBatch = false;
        return true;
    }

    private bool Fail(string message)
    {
        Debug.Log(message);
        processingNextBatch = false;
        return false;
    }

    private void OnError(Exception e)
    {
        isSearching = false;
        filterDropdown.interactable = false; // Disable filtering
        Debug.LogError("Error fetching maps: " + e);
        warning.ShowWarning("Error fetching maps...", Warning.WarningType.Negative);
    }
}
