using UnityEngine;
using Firebase.Auth;
using Firebase.Firestore;
using System;
using System.Collections.Generic;
using System.Collections;

public class Manager_UserData : MonoBehaviour
{
    [SerializeField] private CRUD_Firestore firestoreCRUD;
    [SerializeField] private CRUD_EasySave3_Cooldown cooldown;
    [SerializeField] private Manager_Auth authManager;
    [SerializeField] private Warning warning;
    bool isUploading = false;

    /******* CRUD operation handlers of Map *******/
    
    // Tries to upload a map to Firestore -> C / U
    public void TryUploadPurchases(Dictionary<int, bool> skins, Dictionary<int, bool> trailEffects)
    {
        if (!authManager.IsUserLoggedIn())
        {
            warning.ShowWarning("You need to be logged in to save the purchases!");
            authManager.ShowLogin();
            return;
        }
        if (isUploading)
        {
            return;
        }
        isUploading = true;
        Debug.Log("Uploading purchases to Firestore...");
        firestoreCRUD.UploadPurchasesToFirestore(skins, trailEffects, (success) =>
        {
            if (success)
            {
                Debug.Log("Purchases uploaded successfully.");
            }
            else
            {
                Debug.LogError("Failed to upload purchases.");
            }
            isUploading = false;
        });
    }

    public void LoadPurchases(Action<Dictionary<string, object>> onComplete)
    {
        if (!authManager.IsUserLoggedIn())
        {
            warning.ShowWarning("You need to be logged in to load the purchases...");
            authManager.ShowLogin();
            onComplete?.Invoke(null);
            return;
        }

        if (isUploading)
        {
            Debug.Log("Already loading purchases, skipping duplicate call.");
            return;
        }

        isUploading = true;

        firestoreCRUD.LoadPurchasesFromFirestore(
            purchaseData =>
            {
                Debug.Log("Purchases loaded successfully.");
                isUploading = false;
                onComplete?.Invoke(purchaseData);
            },
            error =>
            {
                Debug.LogError("Failed to load purchases: " + error);
                isUploading = false;
                onComplete?.Invoke(null);
            });
    }

    public void VoteMap(string mapKey, int voteValue, Action<bool> onComplete)
    {
        if (!authManager.IsUserLoggedIn())
        {
            warning.ShowWarning("You need to be logged in to vote for a map.");
            authManager.ShowLogin();
            onComplete?.Invoke(false);
            return;
        }

        if (isUploading)
        {
            Debug.Log("Already voting, skipping duplicate call.");
            return;
        }

        isUploading = true;
        string userID = authManager.GetUserID();

        firestoreCRUD.UpdateUserVote(userID, mapKey, voteValue, success =>
        {
            isUploading = false;
            onComplete?.Invoke(success);
        });
    }

    public void RateMap(string mapKey, int ratingValue, Action<bool, int> onComplete)
    {
        if (!authManager.IsUserLoggedIn())
        {
            warning.ShowWarning("You need to be logged in to rate a map.");
            authManager.ShowLogin();
            onComplete?.Invoke(false, 0);
            return;
        }

        if (isUploading)
        {
            Debug.Log("Already rating, skipping duplicate call.");
            return;
        }

        isUploading = true;
        string userID = authManager.GetUserID();
        
        firestoreCRUD.UpdateUserRate(userID, mapKey, ratingValue, (success, rateCountDelta) =>
        {
            isUploading = false;
            onComplete?.Invoke(success, rateCountDelta);
        });
    }

    public void IncrementMapDownloadCount(string mapKey, Action<bool> onComplete)
    {
        if (isUploading)
        {
            Debug.Log("Already incrementing download count, skipping duplicate call.");
            return;
        }

        isUploading = true;

        firestoreCRUD.IncrementMapDownloads(mapKey, success =>
        {
            isUploading = false;
            onComplete?.Invoke(success);
        });
    }
}
