using UnityEngine;
using System.Collections.Generic;
using System.IO;
using System;

public class CRUD_EasySave3_Cooldown : MonoBehaviour
{
    public class MapInteractionData
    {
        public string lastVoteTime = "";
        public string lastRateTime = "";
        public string lastDownloadTime = "";
        public string lastUploadTime = "";
        public string lastPullTime = "";
    }
    [Serializable]
    public enum MapInteractionType
    {
        Vote,
        Rate,
        Download,
        Upload,
        Pull
    }
    private string FILE_KEY = "interactionData";
    private string FILE_KEY_INDEX = "interactionDataIndex";
    /*** Acces to the file which indexes all the maps ***/
    private Dictionary<string, MapInteractionData> maps = new Dictionary<string, MapInteractionData>();
    
    [SerializeField] private float downloadCD = 1f; // 1 day cooldown for downloading (it is hidden to avoid download increments since it only increments and doesn't require login )
    [SerializeField] private float likeCD = 10f; // 5 seconds cooldown for liking, people may change their mind
    [SerializeField] private float rateCD = 15f; // Rating doesn't change that fast?
    [SerializeField] private float pullCooldown = 30f; // Just to avoid spamming
    [SerializeField] private float uploadCooldown = 60f; // Balance spamming / urgent updates
    [SerializeField] private Interact_MapDataDisplayCloud interactMapCloud;
    [SerializeField] private Warning warning;
    [SerializeField] private MapInteractionType currentInteractionType;

    public void Awake()
    {
        string path = LocalSaveKeys.GetMapPath(FILE_KEY_INDEX, LocalSaveKeys.SaveType.Index);
        if (ES3.FileExists(path))
        {
            maps = ES3.Load<Dictionary<string, MapInteractionData>>(FILE_KEY, path);
        }
        else
        {
            maps = new Dictionary<string, MapInteractionData>();
            ES3.Save(FILE_KEY, maps, path);
        }
    }

    // This functions checks if the CD is over and thus, the interaction can be done
    public string TryUpdateInteraction(string mapName, MapInteractionType interactionType)
    {
        if (!maps.ContainsKey(mapName))
            maps[mapName] = new MapInteractionData();

        // Get the current time
        DateTime now = DateTime.UtcNow;

        // Check last interaction time vs current time and if the cooldown is over --> Use Universal time
        switch (interactionType)
        {
            case MapInteractionType.Vote:
                if (!string.IsNullOrEmpty(maps[mapName].lastVoteTime))
                {
                    DateTime lastVote = DateTime.Parse(
                        maps[mapName].lastVoteTime,
                        null,
                        System.Globalization.DateTimeStyles.AdjustToUniversal
                    );
                    TimeSpan diff = now - lastVote;
                    if (diff.TotalSeconds < likeCD)
                        return FormatCooldown(TimeSpan.FromSeconds(likeCD - diff.TotalSeconds));
                }
                maps[mapName].lastVoteTime = now.ToString("o");
                break;

            case MapInteractionType.Rate:
                if (!string.IsNullOrEmpty(maps[mapName].lastRateTime))
                {
                    DateTime lastRate = DateTime.Parse(
                        maps[mapName].lastRateTime,
                        null,
                        System.Globalization.DateTimeStyles.AdjustToUniversal
                    );
                    TimeSpan diff = now - lastRate;
                    if (diff.TotalSeconds < rateCD)
                        return FormatCooldown(TimeSpan.FromSeconds(rateCD - diff.TotalSeconds));
                }
                maps[mapName].lastRateTime = now.ToString("o");
                break;

            case MapInteractionType.Download:
                if (!string.IsNullOrEmpty(maps[mapName].lastDownloadTime))
                {
                    DateTime lastDownload = DateTime.Parse(
                        maps[mapName].lastDownloadTime,
                        null,
                        System.Globalization.DateTimeStyles.AdjustToUniversal
                    );
                    TimeSpan diff = now - lastDownload;
                    // WARNING: this cooldown is in days, not seconds
                    if (diff.TotalDays < downloadCD)
                        return FormatCooldown(TimeSpan.FromDays(downloadCD - diff.TotalSeconds));
                }
                maps[mapName].lastDownloadTime = now.ToString("o");
                break;

            case MapInteractionType.Upload:
                if (!string.IsNullOrEmpty(maps[mapName].lastUploadTime))
                {
                    DateTime lastUpload = DateTime.Parse(
                        maps[mapName].lastUploadTime,
                        null,
                        System.Globalization.DateTimeStyles.AdjustToUniversal
                    );
                    TimeSpan diff = now - lastUpload;
                    if (diff.TotalSeconds < uploadCooldown)
                        return FormatCooldown(TimeSpan.FromSeconds(uploadCooldown - diff.TotalSeconds));
                }
                maps[mapName].lastUploadTime = now.ToString("o");
                break;
            case MapInteractionType.Pull:
                if (!string.IsNullOrEmpty(maps[mapName].lastPullTime))
                {
                    DateTime lastPull = DateTime.Parse(
                        maps[mapName].lastPullTime,
                        null,
                        System.Globalization.DateTimeStyles.AdjustToUniversal
                    );
                    TimeSpan diff = now - lastPull;
                    if (diff.TotalSeconds < pullCooldown)
                        return FormatCooldown(TimeSpan.FromSeconds(pullCooldown - diff.TotalSeconds));
                }
                maps[mapName].lastPullTime = now.ToString("o");
                break;
        }

        SaveInteractionData();
        return "-1s"; // Success, no cooldown
    }

    public void SetInteractionType(int type)
    {
        currentInteractionType = type switch
        {
            0 => MapInteractionType.Vote,
            1 => MapInteractionType.Rate,
            2 => MapInteractionType.Download,
            3 => MapInteractionType.Upload,
            4 => MapInteractionType.Pull,
            _ => currentInteractionType
        };
    }

    public void UpdateInteraction(GameObject map)
    {
        string mapKey = interactMapCloud.GetMapKey();
        string cooldown = TryUpdateInteraction(mapKey, currentInteractionType);
        if (cooldown != "-1s")
        {
            warning.ShowWarning($"You need to wait {cooldown} before you can do that again.");
            return;
        }
        map.SetActive(true);
    }

    private void SaveInteractionData()
    {
        string path = LocalSaveKeys.GetMapPath(FILE_KEY_INDEX, LocalSaveKeys.SaveType.Index);
        ES3.Save(FILE_KEY, maps, path);
    }

    private string FormatCooldown(TimeSpan span)
    {
        if (span.TotalSeconds < 60)
            return $"{Mathf.CeilToInt((float)span.TotalSeconds)}s";
        else if (span.TotalMinutes < 60)
            return $"{Mathf.CeilToInt((float)span.TotalMinutes)}m";
        else if (span.TotalHours < 24)
            return $"{Mathf.CeilToInt((float)span.TotalHours)}h";
        else
            return $"{Mathf.CeilToInt((float)span.TotalDays)}d";
    }

}