using System;
using Firebase;
using UnityEngine;
using System.Collections.Generic;
using Firebase.Firestore;
using Firebase.Extensions;
using System.Threading.Tasks;
using System.Collections;
using System.Linq;

public class CRUD_Firestore : MonoBehaviour
{
    private FirebaseFirestore firestoreRef = null;
    [SerializeField] private CRUD_Authentication auth;
    [SerializeField] private CRUD_EasySave3 easySave;
    [SerializeField] private CloudMapFetch cloudMapFetch;
    private LocalSaveKeys.SaveType saveType = LocalSaveKeys.SaveType.Download;
    [SerializeField] private bool isUploading = false;
    private bool isDownloading = false;

    private struct BasicInfo
    {
        public string name;
        public string data;
    }

    private BasicInfo mapInfo;

    // Initializes the script
    private void Awake()
    {
        isUploading = false;
        isDownloading = false;
    }
    
    // Sets the Firestore reference
    public void SetFirestoreRef(FirebaseFirestore db)
    {
        firestoreRef = db;
    }

    /********** CRUD operations of Map *******/

   // Uploads a map to Firestore --> C
    public void UploadMapToFirestore(string mapName, Dictionary<string, object> mapDetails, Action<bool> callback)
    {
        Debug.Log("Uploading map to Firestore...");
        if (isUploading || !auth.IsUserLoggedIn())
        {
            Debug.LogWarning("Upload in progress or user not logged in.");
            callback?.Invoke(false);
            return;
        }
        try
        {
            isUploading = true;
            CollectionReference mapsRef = firestoreRef.Collection(FirestoreKeys.SHARED_MAPS_PATH);
            DocumentReference mapDocRef = mapsRef.Document(mapName);

            mapDocRef.SetAsync(mapDetails).ContinueWith(task =>
            {
                isUploading = false;
                if (task.IsFaulted || task.IsCanceled)
                {
                    Debug.LogError("Failed to upload map: " + task.Exception);
                    UnityMainThreadDispatcher.Instance.Enqueue(() => callback?.Invoke(false));
                }
                else
                {
                    Debug.Log("Firestore upload successful!");
                    UnityMainThreadDispatcher.Instance.Enqueue(() =>
                    {
                        Debug.Log("Callback is being invoked on the main thread.");
                        callback?.Invoke(true);
                    });
                }
            });
        }
        catch (Exception ex)
        {
            Debug.LogError("Exception during Firestore upload: " + ex);
            UnityMainThreadDispatcher.Instance.Enqueue(() => callback?.Invoke(false));
        }
    }

 // Updates a map by checking the map in Firestore --> R
public void IsMapUpToDate(string mapID, string lastMod, Action<MapData_Cloud, bool> onUpdateComplete)
{
    if (isUploading) return;

    isUploading = true;

    DocumentReference mapDocRef = firestoreRef.Collection(FirestoreKeys.SHARED_MAPS_PATH).Document(mapID);

    mapDocRef.GetSnapshotAsync().ContinueWithOnMainThread(task =>
    {
        isUploading = false;

        if (task.IsFaulted || task.IsCanceled)
        {
            Debug.LogError("Failed to retrieve map metadata: " + task.Exception);
            onUpdateComplete?.Invoke(null, false);
            return;
        }

        DocumentSnapshot snapshot = task.Result;

        if (!snapshot.Exists)
        {
            Debug.LogError("Map document does not exist: " + mapID);
            onUpdateComplete?.Invoke(null, false);
            return;
        }

        string cloudLastMod = snapshot.GetValue<string>(FirestoreKeys.META_DATA_DATE_MODIFIED);

        bool updateNeeded = cloudLastMod != lastMod;

        Debug.Log($"Cloud date: {cloudLastMod} | Local date: {lastMod} | Update needed: {updateNeeded}");

        // Update the index data locally even if the map is up to date -> so it loads the correct metadata not only the map data
        MapData_Cloud updatedMapData = null;
        Dictionary<string, object> mapData = snapshot.ToDictionary();
            // Create updated MapData_Cloud
        updatedMapData = cloudMapFetch.CreateMapDataFromFirestore(mapData);

        // Return the updated map data (if any) and whether an update was needed
        onUpdateComplete?.Invoke(updatedMapData, updateNeeded);
    });

    isUploading = false;
}


    public void DeleteMap(string mapID, Action<bool> onDeleteComplete)
    {
        DeleteMapFromFirestore(mapID).ContinueWithOnMainThread(task =>
        {
            bool success = !task.IsFaulted && !task.IsCanceled;
            onDeleteComplete?.Invoke(success);
        });
    }

    private async Task DeleteMapFromFirestore(string mapID)
    {
        DocumentReference mapDocRef = firestoreRef.Collection(FirestoreKeys.SHARED_MAPS_PATH).Document(mapID);
        await mapDocRef.DeleteAsync();
        Debug.Log("Map Deleted");
    }

    /*************** User related methods ****************/

    // Saves the short base62 hash of the user ID -> C
    public void SaveShortBase62Hash(string shortBase62Hash, string userId, Action onSuccess, Action<Exception> onError)
    {
        DocumentReference docRef = firestoreRef.Collection(FirestoreKeys.USER_MAPS_PATH).Document(shortBase62Hash);
        docRef.SetAsync(new { shortBase62Hash = userId }).ContinueWithOnMainThread(task =>
        {
            if (task.IsCanceled || task.IsFaulted)
            {
                onError?.Invoke(task.Exception);
                return;
            }
            onSuccess?.Invoke();
        });
    }

    // Finds the user ID using the short base62 hash -> R
    public void FindUserID(string shortID, Action<DocumentSnapshot> onSuccess, Action<Exception> onError)
    {
        DocumentReference docRef = firestoreRef.Collection(FirestoreKeys.USER_MAPS_PATH).Document(shortID);
        docRef.GetSnapshotAsync().ContinueWithOnMainThread(task =>
        {
            if (task.IsCanceled || task.IsFaulted)
            {
                onError?.Invoke(task.Exception);
                return;
            }
            onSuccess?.Invoke(task.Result);
        });
    }

     // Finds the user ID using the short base62 hash -> R
    public void UploadPurchasesToFirestore(Dictionary<int, bool> skins, Dictionary<int, bool> trailEffects, Action<bool> callback)
    {
        if (!auth.IsUserLoggedIn())
        {
            Debug.LogWarning("User is not logged in. Cannot upload purchases.");
            callback?.Invoke(false);
            return;
        }
        DocumentReference docRef = firestoreRef.Collection(FirestoreKeys.USER_MAPS_PATH).Document(auth.GetUserID());
        
        // Convert int-keyed dictionaries to string-keyed dictionaries
        Dictionary<string, object> skinsStrKeys = skins.ToDictionary(
            kvp => kvp.Key.ToString(),
            kvp => (object)kvp.Value
        );

        Dictionary<string, object> trailEffectsStrKeys = trailEffects.ToDictionary(
            kvp => kvp.Key.ToString(),
            kvp => (object)kvp.Value
        );

        // Create a dictionary to hold the purchase data
        Dictionary<string, object> purchaseData = new Dictionary<string, object>
        {
            { ShopKeys.SKIN_FILE, skinsStrKeys },
            { ShopKeys.TRAIL_FILE, trailEffectsStrKeys }
        };

        docRef.SetAsync(purchaseData).ContinueWithOnMainThread(task =>
        {
            if (task.IsFaulted || task.IsCanceled)
            {
                Debug.LogError("Failed to upload purchases: " + task.Exception);
                callback?.Invoke(false);
            }
            else
            {
                Debug.Log("Purchases uploaded successfully!");
                callback?.Invoke(true);
            }
        });
    }

    public void LoadPurchasesFromFirestore(
        Action<Dictionary<string, object>> onSuccess,
        Action<Exception> onError)
    {
        if (!auth.IsUserLoggedIn())
        {
            Debug.LogWarning("User is not logged in. Cannot upload purchases.");
            onSuccess?.Invoke(null);
            return;
        }

        DocumentReference docRef = firestoreRef
            .Collection(FirestoreKeys.USER_MAPS_PATH)
            .Document(auth.GetUserID());

        docRef.GetSnapshotAsync().ContinueWithOnMainThread(task =>
        {
            if (task.IsFaulted || task.IsCanceled)
            {
                onError?.Invoke(task.Exception);
                return;
            }

            DocumentSnapshot snapshot = task.Result;
            var purchaseData = snapshot.ToDictionary();
            
            if (purchaseData.ContainsKey(ShopKeys.SKIN_FILE) && purchaseData[ShopKeys.SKIN_FILE] is Dictionary<string, object> skins &&
            purchaseData.ContainsKey(ShopKeys.TRAIL_FILE) && purchaseData[ShopKeys.TRAIL_FILE] is Dictionary<string, object> trails)
            {
                onSuccess?.Invoke(purchaseData);
            }
            else
            {
                Debug.Log("No purchases found for the user.");
                onSuccess?.Invoke(null); // or an empty dictionary
            }
        });
    }

    public void UpdateUserVote(string userID, string mapID, int newVote, Action<bool> onComplete)
    {
        if (newVote != 1 && newVote != -1)
        {
            Debug.LogError("Invalid vote value. Must be 1 or -1.");
            onComplete?.Invoke(false);
            return;
        }

        DocumentReference mapDocRef = firestoreRef
            .Collection(FirestoreKeys.USER_MAPS_PATH)
            .Document(userID)
            .Collection(FirestoreKeys.USER_INTERACTED_MAPS)
            .Document(mapID);

        mapDocRef.GetSnapshotAsync().ContinueWithOnMainThread(task =>
        {
            if (task.IsFaulted || task.IsCanceled)
            {
                Debug.LogError("Failed to read user vote: " + task.Exception);
                onComplete?.Invoke(false);
                return;
            }

            DocumentSnapshot snapshot = task.Result;
            int? oldVote = null;

            if (snapshot.Exists && snapshot.ContainsField(FirestoreKeys.META_DATA_LIKES))
            {
                oldVote = snapshot.GetValue<int>(FirestoreKeys.META_DATA_LIKES);
            }

            int voteDelta;
            if (oldVote == null)
            {
                voteDelta = newVote;
            }
            else if (oldVote == newVote)
            {
                onComplete?.Invoke(true);
                return;
            }
            else
            {
                voteDelta = newVote - oldVote.Value;
            }

            Dictionary<string, object> updateData = new Dictionary<string, object>
            {
                { FirestoreKeys.META_DATA_LIKES, newVote }
            };

            mapDocRef.SetAsync(updateData, SetOptions.MergeAll).ContinueWithOnMainThread(updateTask =>
            {
                if (updateTask.IsFaulted || updateTask.IsCanceled)
                {
                    onComplete?.Invoke(false);
                    return;
                }

                // Update global count only if personal vote update succeeded
                VoteMap(mapID, voteDelta, success =>
                {
                    onComplete?.Invoke(success);
                });
            });
        });
    }

    public void VoteMap(string mapID, int voteValue, Action<bool> onVoteComplete)
    {
        DocumentReference mapDocRef = firestoreRef.Collection(FirestoreKeys.SHARED_MAPS_PATH).Document(mapID);

        Dictionary<string, object> updateData = new Dictionary<string, object>
        {
            { FirestoreKeys.META_DATA_LIKES, FieldValue.Increment(voteValue) }
        };

        mapDocRef.UpdateAsync(updateData).ContinueWithOnMainThread(task =>
        {
            bool success = !task.IsFaulted && !task.IsCanceled;
            onVoteComplete?.Invoke(success);
        });
    }

    public void UpdateUserRate(string userID, string mapID, int newRate, Action<bool, int> onComplete)
    {
        if (newRate < 1 || newRate > 5)
        {
            Debug.LogError("Invalid rate value. Must be between 1 and 5.");
            onComplete?.Invoke(false, 0);
            return;
        }

        DocumentReference mapDocRef = firestoreRef
            .Collection(FirestoreKeys.USER_MAPS_PATH)
            .Document(userID)
            .Collection(FirestoreKeys.USER_INTERACTED_MAPS)
            .Document(mapID);

        mapDocRef.GetSnapshotAsync().ContinueWithOnMainThread(task =>
        {
            if (task.IsFaulted || task.IsCanceled)
            {
                Debug.LogError("Failed to read user rating: " + task.Exception);
                onComplete?.Invoke(false, 0);
                return;
            }

            DocumentSnapshot snapshot = task.Result;
            int? oldRate = null;

            if (snapshot.Exists && snapshot.ContainsField("rate"))
            {
                oldRate = snapshot.GetValue<int>("rate");
            }

            int rateDelta;
            bool isNewRating;

            if (oldRate == null)
            {
                rateDelta = newRate;
                isNewRating = true;
            }
            else if (oldRate == newRate)
            {
                onComplete?.Invoke(true, 0);
                return;
            }
            else
            {
                rateDelta = newRate - oldRate.Value;
                isNewRating = false;
            }

            Dictionary<string, object> updateData = new Dictionary<string, object>
            {
                { "rate", newRate }
            };

            mapDocRef.SetAsync(updateData, SetOptions.MergeAll).ContinueWithOnMainThread(updateTask =>
            {
                if (updateTask.IsFaulted || updateTask.IsCanceled)
                {
                    onComplete?.Invoke(false, 0);
                    return;
                }

                RateMap(mapID, rateDelta, isNewRating, rateSuccess =>
                {
                    onComplete?.Invoke(rateSuccess, isNewRating ? 1 : 0);
                });
            });
        });
    }

    private void RateMap(string mapID, int rateDelta, bool isNewRating, Action<bool> onRateComplete)
    {
        DocumentReference mapDocRef = firestoreRef.Collection(FirestoreKeys.SHARED_MAPS_PATH).Document(mapID);

        Dictionary<string, object> updateData = new Dictionary<string, object>
        {
            { FirestoreKeys.META_DATA_RATING, FieldValue.Increment(rateDelta) }
        };

        if (isNewRating)
        {
            // Only increment rating count if it's the first time the user rated
            updateData[FirestoreKeys.META_DATA_RATES] = FieldValue.Increment(1);
        }

        mapDocRef.UpdateAsync(updateData).ContinueWithOnMainThread(task =>
        {
            bool success = !task.IsFaulted && !task.IsCanceled;
            onRateComplete?.Invoke(success);
        });
    }

    public void IncrementMapDownloads(string mapID, Action<bool> onComplete)
    {
        DocumentReference mapDocRef = firestoreRef
            .Collection(FirestoreKeys.SHARED_MAPS_PATH)
            .Document(mapID);

        Dictionary<string, object> updateData = new Dictionary<string, object>
        {
            { FirestoreKeys.META_DATA_DOWNLOADS, FieldValue.Increment(1) }
        };

        mapDocRef.UpdateAsync(updateData).ContinueWithOnMainThread(task =>
        {
            bool success = !task.IsFaulted && !task.IsCanceled;
            if (!success)
                Debug.LogError("Failed to increment map downloads: " + task.Exception);
            
            onComplete?.Invoke(success);
        });
    }

}

