using System;
using Firebase;
using UnityEngine;
using System.Collections.Generic;
using Firebase.Storage;
using Firebase.Extensions;
using System.Threading.Tasks;
using System.Collections;
using System.IO;
using UnityEngine.Networking;

// CRUD_Firestore will require refactoring as storage now takes care of some of the operations
public class CRUD_Storage : MonoBehaviour
{
  private FirebaseStorage storage = null;
  [SerializeField] private CRUD_Authentication auth;
  [SerializeField] private Interact_MapDataDisplayLocal mapHolder;
  [SerializeField] private CRUD_EasySave3 easySave;
  private LocalSaveKeys.SaveType saveType = LocalSaveKeys.SaveType.Download;
  private bool isUploading = false;
  private bool isDownloading = false;

  private struct BasicInfo
  {
    public string name;
    public string data;
  }

  private BasicInfo mapInfo;

  // Initializes the script
  private void Awake()
  {
    isUploading = false;
    isDownloading = false;
  }
  
  // Sets the Firestore reference
  public void SetStorageRef(FirebaseStorage defInstance)
  {
    storage = defInstance;
  }

  /********** CRUD operations of Map *******/

  // Uploads a map to Firestore --> C
  public void UploadMapData(string mapKey, string mapTilesData, Action<string> callback)
  {
    if (isUploading) return;
    isUploading = true;
    // Convert JSON string to byte array
    byte[] jsonData = System.Text.Encoding.UTF8.GetBytes(mapTilesData);

    // Initialize Firebase Storage reference
    StorageReference storageRef = storage.GetReference($"map_tiles_data/{mapKey}.json");

    // Upload data directly using PutBytesAsync
    storageRef.PutBytesAsync(jsonData).ContinueWith(uploadTask =>
    {
        if (uploadTask.IsFaulted || uploadTask.IsCanceled)
        {
            Debug.LogError("Upload failed: " + uploadTask.Exception);
            callback?.Invoke(uploadTask.Exception.ToString());
            isUploading = false;
            return;
        }

        // 4. Get download URL after successful upload
        storageRef.GetDownloadUrlAsync().ContinueWith(urlTask =>
        {
            if (urlTask.IsFaulted || urlTask.IsCanceled)
            {
                Debug.LogError("URL retrieval failed: " + urlTask.Exception);
                callback?.Invoke(urlTask.Exception.ToString());
            }
            else
            {
                callback?.Invoke(urlTask.Result.ToString());
            }
        });
        
    });

    isUploading = false;
  }

    // Reads the map data from Firestore --> R
    public void DownloadMap(string url, string mapKey, Action onSuccess, Action<Exception> onError)
    {
        if (isDownloading) return;

        isDownloading = true;
        // Start downloading from Firebase Storage
        StartCoroutine(DownloadFileFromFirebase(url, mapKey, onSuccess, onError));
    }

    private IEnumerator DownloadFileFromFirebase(string url, string mapKey, Action onSuccess, Action<Exception> onError)
    {
        using (UnityEngine.Networking.UnityWebRequest request = UnityEngine.Networking.UnityWebRequest.Get(url))
        {
            yield return request.SendWebRequest();

            if (request.result == UnityEngine.Networking.UnityWebRequest.Result.Success)
            {
                try
                {
                    // Convert byte[] to string "Some techinal debt here"
                    string jsonData = System.Text.Encoding.UTF8.GetString(request.downloadHandler.data);
                    
                    // Save as string (instead of byte[])
                    easySave.SaveRawMapData(mapKey, jsonData, saveType);
                    Debug.Log($"File successfully downloaded and saved");
                    onSuccess?.Invoke();
                }
                catch (Exception ex)
                {
                    Debug.LogError("Failed to write file: " + ex);
                    onError?.Invoke(ex);
                }
            }
            else
            {
                Debug.LogError("Download failed: " + request.error);
                onError?.Invoke(new Exception(request.error));
            }
        }

        isDownloading = false;
    }

    public void DeleteMap(string mapKey, Action<bool> onDeleteResult)
    {
        if (isUploading) return;

        isUploading = true;
        StorageReference storageRef = storage.GetReference($"map_tiles_data/{mapKey}.json");
        storageRef.DeleteAsync().ContinueWith(task =>
        {
            if (task.IsFaulted || task.IsCanceled)
            {
                Debug.LogError("Failed to delete map: " + task.Exception);
                onDeleteResult?.Invoke(false);
            }
            else
            {
                Debug.Log("Map deleted successfully.");
                onDeleteResult?.Invoke(true);
            }
        });

        isUploading = false;
    }

     /*********** Other CRUD accesses ************/

    /*** Map data handling ***/
    private void SetMapInfo(string name, string data)
    {
        mapInfo.name = name;
        mapInfo.data = data;
    }


}
