using UnityEngine;
using System;
using System.Collections.Generic;
using Firebase.Firestore;
using Firebase.Extensions;
using System.Linq;
using System.Collections;
using System.Globalization;


public class CloudMapFetch : MonoBehaviour
{
    private List<DocumentSnapshot> snapshot;
    private int currentIndex = 0;
    private int batchSize = 5;
    [SerializeField] private bool isQuerying = false;
    [SerializeField] private bool isDownloading = false;

    [SerializeField] private FirebaseFirestore firestoreRef;

    // Sets the Firestore reference
    public void SetFirestoreRef(FirebaseFirestore db)
    {
        firestoreRef = db;
    }

    // Checks if Firestore is connected
    public bool IsFirestoreConnected()
    {
        return firestoreRef != null;
    }

    // Processes the next batch of maps
    public void ProcessNextBatch(Action<List<MapData_Cloud>> onSuccess, int batchSize)
    {
        if (snapshot == null || currentIndex >= snapshot.Count)
        {
            onSuccess?.Invoke(new List<MapData_Cloud>()); // No more data
            return;
        }

        List<MapData_Cloud> maps = new List<MapData_Cloud>();

        for (int i = 0; i < batchSize && currentIndex < snapshot.Count; i++, currentIndex++)
        {
            DocumentSnapshot document = snapshot[currentIndex];
            if (document.Exists)
            {
                Dictionary<string, object> mapData = document.ToDictionary();
                MapData_Cloud cloudMap = CreateMapDataFromFirestore(mapData);
                maps.Add(cloudMap);
            }
        }

        onSuccess?.Invoke(maps); // Return this batch
    }

    public MapData_Cloud CreateMapDataFromFirestore(Dictionary<string, object> mapData)
    {
        // Extract basic fields
        string _key = mapData.ContainsKey(FirestoreKeys.MAP_KEY) ? mapData[FirestoreKeys.MAP_KEY].ToString() : "Unknown";
        string _mapName = mapData.ContainsKey(FirestoreKeys.MAP_NAME) ? mapData[FirestoreKeys.MAP_NAME].ToString() : "Unknown";
        string _mapAuthor = mapData.ContainsKey(FirestoreKeys.MAP_AUTHOR) ? mapData[FirestoreKeys.MAP_AUTHOR].ToString() : "Unknown";
        string _mapAuthorShortID = mapData.ContainsKey(FirestoreKeys.MAP_AUTHOR_ID) ? mapData[FirestoreKeys.MAP_AUTHOR_ID].ToString() : "Unknown";
        string _mapDescription = mapData.ContainsKey(FirestoreKeys.MAP_DESCRIPTION) ? mapData[FirestoreKeys.MAP_DESCRIPTION].ToString() : "No description";
        string _lastModifiedDate = mapData.ContainsKey(FirestoreKeys.META_DATA_DATE_MODIFIED) ? mapData[FirestoreKeys.META_DATA_DATE_MODIFIED].ToString() : DateTimeGenerator.GetCurrentDateTime();
        string _creationDate = mapData.ContainsKey(FirestoreKeys.META_DATA_DATE_CREATED) ? mapData[FirestoreKeys.META_DATA_DATE_CREATED].ToString() : DateTimeGenerator.GetCurrentDateTime();
        string _mapData = mapData.ContainsKey(FirestoreKeys.MAP_DATA) ? mapData[FirestoreKeys.MAP_DATA].ToString() : "No data";
        int _downloads = mapData.ContainsKey(FirestoreKeys.META_DATA_DOWNLOADS) ? Convert.ToInt32(mapData[FirestoreKeys.META_DATA_DOWNLOADS]) : 0;
        int _likes = mapData.ContainsKey(FirestoreKeys.META_DATA_LIKES) ? Convert.ToInt32(mapData[FirestoreKeys.META_DATA_LIKES]) : 0;
        int _rating = mapData.ContainsKey(FirestoreKeys.META_DATA_RATING) ? Convert.ToInt32(mapData[FirestoreKeys.META_DATA_RATING]) : 0;
        int _rates = mapData.ContainsKey(FirestoreKeys.META_DATA_RATES) ? Convert.ToInt32(mapData[FirestoreKeys.META_DATA_RATES]) : 0;
        string _fileSize = mapData.ContainsKey(FirestoreKeys.META_DATA_FILE_SIZE) ? mapData[FirestoreKeys.META_DATA_FILE_SIZE].ToString() : "??? MB";
        
        // Create the MapData_Cloud object
        MapData_Cloud mapDataCloud = new MapData_Cloud()
        {
            key = _key,
            mapName = _mapName,
            mapAuthor = _mapAuthor,
            mapAuthorID = _mapAuthorShortID,
            mapDescription = _mapDescription,
            lastModifiedDate = _lastModifiedDate,
            creationDate = _creationDate,
            mapDataPath = _mapData,
            downloads = _downloads,
            likes = _likes,
            totalRating = _rating,
            rates = _rates,
            fileSize = _fileSize
        };

        // Deserialize holesPar
        if (mapData.ContainsKey(FirestoreKeys.MAP_HOLES))
        {
            Dictionary<string, object> firestoreHolesPar = mapData[FirestoreKeys.MAP_HOLES] as Dictionary<string, object>;
            if (firestoreHolesPar != null)
            {
                foreach (var holeEntry in firestoreHolesPar)
                {
                    if (int.TryParse(holeEntry.Key, out int holeKey))
                    {
                        Dictionary<string, object> holeData = holeEntry.Value as Dictionary<string, object>;
                        if (holeData != null)
                        {
                            int parValue = holeData.ContainsKey("parValue") ? Convert.ToInt32(holeData["parValue"]) : 1;
                            float posX = holeData.ContainsKey("posX") ? Convert.ToSingle(holeData["posX"]) : 0.5f;
                            float posY = holeData.ContainsKey("posY") ? Convert.ToSingle(holeData["posY"]) : -4.5f;

                            mapDataCloud.holesPar[holeKey] = new HoleInfo(parValue, posX, posY);
                        }
                    }
                }
            }
        }

        return mapDataCloud;
    }

    
    // Get the maps by the author's short IDpublic void GetMapByID(string mapID, Action<MapData_Cloud> onSuccess, Action<Exception> onError)
    public void GetMapByID(string mapID, Action<List<MapData_Cloud>> onSuccess, Action<Exception> onError)
    {
        if (isQuerying) return;

        isQuerying = true;

        CollectionReference mapsRef = firestoreRef.Collection(FirestoreKeys.SHARED_MAPS_PATH);

        mapsRef
            .WhereEqualTo(FirestoreKeys.MAP_KEY, mapID)
            .Limit(1) // Since we're querying by ID, there should only be one result
            .GetSnapshotAsync()
            .ContinueWithOnMainThread(task =>
            {
                isQuerying = false;
                if (task.IsCompleted && !task.IsFaulted && !task.IsCanceled)
                {
                    snapshot = task.Result.Documents.ToList();
                    currentIndex = 0;
                    ProcessNextBatch(onSuccess, batchSize); // Process the first batch
                }
                else
                {
                    onError?.Invoke(task.Exception);
                }
            });
    }

    public void GetMapsByAuthorID(string authorID, Action<List<MapData_Cloud>> onSuccess, Action<Exception> onError, int batchSize = 10)
    {
        Debug.Log("Searching for maps by author ID: " + authorID);
        if (isQuerying) return;

        isQuerying = true;

        CollectionReference mapsRef = firestoreRef.Collection(FirestoreKeys.SHARED_MAPS_PATH);

        mapsRef
            .WhereEqualTo(FirestoreKeys.MAP_AUTHOR_ID, authorID)
            .GetSnapshotAsync()
            .ContinueWithOnMainThread(task =>
            {
                Debug.Log("Maps by author ID query completed.");
                isQuerying = false;
                if (task.IsCompleted && !task.IsFaulted && !task.IsCanceled)
                {
                    snapshot = task.Result.Documents.ToList();
                    currentIndex = 0;
                    ProcessNextBatch(onSuccess, batchSize); // Process the first batch
                }
                else
                {
                    onError?.Invoke(task.Exception);
                }
            });
    }

    public void GetMapsByName(string mapNamePrefix, Action<List<MapData_Cloud>> onSuccess, Action<Exception> onError, int batchSize = 10)
    {
        if (isQuerying) return;

        isQuerying = true;

        mapNamePrefix = mapNamePrefix.ToLower(); // Ensure the prefix is in lower case

        CollectionReference mapsRef = firestoreRef.Collection(FirestoreKeys.SHARED_MAPS_PATH);

        string startAt = mapNamePrefix;
        string endAt = mapNamePrefix + "\uf8ff";

        mapsRef
            .WhereGreaterThanOrEqualTo(FirestoreKeys.MAP_LOWER_CASE_NAME, startAt)
            .WhereLessThan(FirestoreKeys.MAP_LOWER_CASE_NAME, endAt)
            .OrderBy(FirestoreKeys.MAP_LOWER_CASE_NAME) // Ensure consistent ordering
            .GetSnapshotAsync()
            .ContinueWithOnMainThread(task =>
            {
                isQuerying = false;
                if (task.IsCompleted && !task.IsFaulted && !task.IsCanceled)
                {
                    snapshot = task.Result.Documents.ToList();
                    currentIndex = 0;
                    ProcessNextBatch(onSuccess, batchSize); // Process the first batch
                }
                else
                {
                    onError?.Invoke(task.Exception);
                }
            });
    }

    // Get the most recet maps
    public void GetMostRecentMaps(Action<List<MapData_Cloud>> onSuccess, Action<Exception> onError, int batchSize = 10)
    {
        if (isQuerying) return;

        isQuerying = true;

        CollectionReference mapsRef = firestoreRef.Collection(FirestoreKeys.SHARED_MAPS_PATH);

        mapsRef
            .GetSnapshotAsync()
            .ContinueWithOnMainThread(task =>
            {
                isQuerying = false;

                if (task.IsCompleted && !task.IsFaulted && !task.IsCanceled)
                {
                    try
                    {
                        var documents = task.Result.Documents.ToList();

                        // Sort the documents based on the creation date (META_DATA_DATE_CREATED)
                        var sortedDocs = documents
                            .Where(doc => doc.ContainsField(FirestoreKeys.META_DATA_DATE_CREATED))
                            .OrderBy(doc =>
                            {
                                var dateStr = doc.GetValue<string>(FirestoreKeys.META_DATA_DATE_CREATED);
                                DateTime parsedDate = DateTime.ParseExact(dateStr, "dd-MM-yyyy HH:mm:ss", CultureInfo.InvariantCulture);
                                return parsedDate;
                            })
                            .ToList();

                        // Store the sorted documents in snapshot
                        snapshot = sortedDocs;
                        currentIndex = 0;

                        ProcessNextBatch(onSuccess, batchSize);
                    }
                    catch (Exception ex)
                    {
                        onError?.Invoke(ex);
                    }
                }
                else
                {
                    onError?.Invoke(task.Exception);
                }
            });
    }

    // Get the most downloaded maps
    public void GetMostDownloadedMaps(Action<List<MapData_Cloud>> onSuccess, Action<Exception> onError, int batchSize = 10)
    {
        if (isQuerying) return;

        isQuerying = true;

        CollectionReference mapsRef = firestoreRef.Collection(FirestoreKeys.SHARED_MAPS_PATH);

        mapsRef
            .OrderBy(FirestoreKeys.META_DATA_DOWNLOADS)
            .GetSnapshotAsync()
            .ContinueWithOnMainThread(task =>
            {
                isQuerying = false;

                if (task.IsCompleted && !task.IsFaulted && !task.IsCanceled)
                {
                    snapshot = task.Result.Documents.ToList();
                    currentIndex = 0;
                    ProcessNextBatch(onSuccess, batchSize);
                }
                else
                {
                    onError?.Invoke(task.Exception);
                }
            });
    }
    // Get the most liked maps
    public void GetMostLikedMaps(Action<List<MapData_Cloud>> onSuccess, Action<Exception> onError, int batchSize = 10)
    {
        if (isQuerying) return;

        isQuerying = true;

        CollectionReference mapsRef = firestoreRef.Collection(FirestoreKeys.SHARED_MAPS_PATH);

        mapsRef
            .OrderBy(FirestoreKeys.META_DATA_LIKES)
            .GetSnapshotAsync()
            .ContinueWithOnMainThread(task =>
            {
                isQuerying = false;

                if (task.IsCompleted && !task.IsFaulted && !task.IsCanceled)
                {
                    snapshot = task.Result.Documents.ToList();
                    currentIndex = 0;
                    ProcessNextBatch(onSuccess, batchSize);
                }
                else
                {
                    onError?.Invoke(task.Exception);
                }
            });
    }

    // Get the best rated maps
    public void GetBestRatedMaps(Action<List<MapData_Cloud>> onSuccess, Action<Exception> onError, int batchSize = 10)
    {
        if (isQuerying) return;
    isQuerying = true;

    CollectionReference mapsRef = firestoreRef.Collection(FirestoreKeys.SHARED_MAPS_PATH);

    mapsRef.GetSnapshotAsync()
        .ContinueWithOnMainThread(task =>
        {
            isQuerying = false;

            if (task.IsCompleted && !task.IsFaulted && !task.IsCanceled)
            {
                try
                {
                    var documents = task.Result.Documents;

                    // Filter documents to check for specific fields
                    var sortedDocs = documents
                        .Where(doc => doc.ContainsField(FirestoreKeys.META_DATA_RATES) &&
                                    doc.ContainsField(FirestoreKeys.META_DATA_RATING))  // Use ContainsField instead of Contains
                        .Where(doc => Convert.ToInt32(doc.GetValue<long>(FirestoreKeys.META_DATA_RATES)) > 0)
                        .OrderBy(doc =>
                        {
                            int total = Convert.ToInt32(doc.GetValue<long>(FirestoreKeys.META_DATA_RATING));
                            int rates = Convert.ToInt32(doc.GetValue<long>(FirestoreKeys.META_DATA_RATES));
                            return (float)total / rates;  // Calculate average rating
                        })
                        .ThenBy(doc => Convert.ToInt32(doc.GetValue<long>(FirestoreKeys.META_DATA_RATES)))  // Sort by rating count
                        .ToList();

                    // Store sorted documents in snapshot
                    snapshot = sortedDocs;  // Make sure snapshot is a List<DocumentSnapshot> instead of QuerySnapshot
                    currentIndex = 0;

                    ProcessNextBatch(onSuccess, batchSize);
                }
                catch (Exception ex)
                {
                    onError?.Invoke(ex);
                }
            }
            else
            {
                onError?.Invoke(task.Exception);
            }
        });
    }
}

