using System.Collections.Generic;
using UnityEngine;

public class UnityMainThreadDispatcher : MonoBehaviour
{
  private static readonly Queue<System.Action> _executionQueue = new Queue<System.Action>();

  private void Awake()
  {
    if (_instance == null)
    {
      _instance = this;
      DontDestroyOnLoad(gameObject); // Make it persistent
    }
    else
    {
      Destroy(gameObject); // Ensure only one instance exists
    }
  }
  void Start()
  {
    UnityMainThreadDispatcher.Instance.Enqueue(() =>
    {
      Debug.Log("This should run on the main thread.");
    });
  }

  public void Update()
  {
    lock (_executionQueue)
    {
      while (_executionQueue.Count > 0)
      {
        try
        {
          Debug.Log("Executing action on the main thread.");
          _executionQueue.Dequeue().Invoke();
        }
        catch (System.Exception ex)
        {
          Debug.LogError("Error executing action on the main thread: " + ex);
        }
      }
    }
  }

  public void Enqueue(System.Action action)
  {
    lock (_executionQueue)
    {
      Debug.Log("Action enqueued for execution on the main thread.");
      _executionQueue.Enqueue(action);
    }
  }

  private static UnityMainThreadDispatcher _instance;
  public static UnityMainThreadDispatcher Instance
  {
    get
    {
      if (_instance == null)
      {
        _instance = FindObjectOfType<UnityMainThreadDispatcher>();
        if (_instance == null)
        {
          Debug.Log("Creating UnityMainThreadDispatcher instance.");
          GameObject dispatcher = new GameObject("UnityMainThreadDispatcher");
          _instance = dispatcher.AddComponent<UnityMainThreadDispatcher>();
        }
      }
      return _instance;
    }
  }
}