using UnityEngine;
using TMPro;
using Unity.Services.Core;
using Unity.Services.Lobbies;
using Unity.Services.Lobbies.Models;
public class Manager_Lobby : MonoBehaviour
{
  [SerializeField] private UI_LobbyData uiLobbyData;
  [SerializeField] private UI_LobbyView uiLobbyView;
  [SerializeField] private UI_LobbyData.LobbyData lobbyData;
  [SerializeField] private Mult_Lobby multLobby;
  [SerializeField] private Interact_LobbyPlayerDisplay interactLobbyPlayerDisplay;
  [SerializeField] private Warning warning;

  // If it gets bigger, we can move it to a separate class
  [SerializeField] private TMP_InputField lobbyCodeInput;
  [SerializeField] private GameObject codeInputPanel;
  private bool isOnLobby = false;

  private void Start()
  {
    uiLobbyData.createButton.onClick.AddListener(CreateLobby);
    OnEnable();
  }

  private void OnEnable()
  {
    // Subscribe to the event
    multLobby.OnLobbyUpdated += HandleLobbyUpdated;
  }

  private void OnDisable()
  {
    // Unsubscribe from the event when the script is disabled
    multLobby.OnLobbyUpdated -= HandleLobbyUpdated;
  }

  // This method is called when the event is triggered
  private void HandleLobbyUpdated(Lobby updatedLobby)
  {
    if (isOnLobby)
    {
      UpdateLobbyView(updatedLobby);
    }
  }

  // Main interaction methods ************************************************************************************************

  // Create a lobby and populate the lobby menu / create a network lobby
  private async void CreateLobby()
  {
    lobbyData = uiLobbyData.GetLobbyData();

    // Create Network Lobby and await its completion
    await multLobby.CreateLobby(lobbyData);

    // Now that the lobby is created, retrieve it
    Lobby lobby = multLobby.GetClientLobby();

    UpdateLobbyView(lobby);
  }

  public async void JoinServerByCode()
  {
    string lobbyCode = lobbyCodeInput.text;
    try
    {
      // Join the server
      Lobby lobby = await multLobby.JoinLobbyByCode(lobbyCode);
      UpdateLobbyView(lobby);
      codeInputPanel.SetActive(false);
    }
    catch (LobbyServiceException e)
    {
      warning.ShowWarning("Error: " + e.Message, Warning.WarningType.Negative);
      Debug.LogError("Failed to join the server: " + e.Message);
    }
  }

  // Join by id
  public async void JoinServerById(string serverId)
  {
    try
    {
      // Join the server
      Lobby lobby = await multLobby.JoinLobbyById(serverId);
      UpdateLobbyView(lobby);
    }
    catch (LobbyServiceException e)
    {
      warning.ShowWarning("Error: " + e.Message, Warning.WarningType.Negative);
      Debug.LogError("Failed to join the server: " + e.Message);
    }
  }


  // Leave the lobby
  public void LeaveLobby()
  {
    if (multLobby.GetClientLobby() == null)
    {
      Debug.LogError("No lobby to leave.");
      return;
    }
    OnDisable();
    multLobby.LeaveLobby();
    interactLobbyPlayerDisplay.ClearPlayersDisplay();
    uiLobbyView.HideLobbyView();
    isOnLobby = false;
  }

  // Support methods ************************************************************************************************
  private void GetLobbyData(Lobby lobby)
  {
    lobbyData = new UI_LobbyData.LobbyData();
    lobbyData.lobbyName = lobby.Name;
    lobbyData.isPrivate = lobby.IsPrivate;
    lobbyData.maxPlayers = lobby.MaxPlayers;
    lobbyData.mapName = lobby.Data["MapName"].Value;
    lobbyData.mapId = lobby.Data["MapId"].Value;
  }
  // Update the players in the lobby
  public void UpdateLobbyPlayers(Lobby lobby, string  hostID)
  {
    // Clear the players before updating
    interactLobbyPlayerDisplay.ClearPlayers();
    interactLobbyPlayerDisplay.ClearPlayersDisplay();

    foreach (Player player in lobby.Players)
    {
      // Create the player object
      PlayerData hostPlayer = new PlayerData();
      hostPlayer.name = player.Data["PlayerName"].Value;
      hostPlayer.id = player.Id;
      hostPlayer.isReady = false;
      hostPlayer.isHost = player.Id.Equals(hostID);

      // Add the player to the list
      interactLobbyPlayerDisplay.AddPlayer(hostPlayer);
    }
    interactLobbyPlayerDisplay.LoadPlayers();
  }

  // Populate the lobby menu
  private void PopulateLobbyMenu(UI_LobbyData.LobbyData lobbyData, int playerNumber)
  {
    uiLobbyView.ShowLobbyView();
    uiLobbyView.SetLobbyName(lobbyData.lobbyName);
    uiLobbyView.SetLobbyPlayerCount(playerNumber, lobbyData.maxPlayers);
  }

  // Update the lobby view data
  private void UpdateLobbyView(Lobby lobby)
  {
    Debug.Log("Lobby updated: " + lobby.Name);
    if (lobby != null)
    {
      isOnLobby = true;
      // Get the host ID from the updated lobby
      string hostID = lobby.HostId;

      // Update the lobby players
      UpdateLobbyPlayers(lobby, hostID);

      // Set the lobby ID code
      uiLobbyView.SetLobbyId(lobby.LobbyCode);

      // Get the lobby data
      GetLobbyData(lobby);

      int playerNumber = lobby.Players.Count;

      // Populate locally the lobby menu
      PopulateLobbyMenu(lobbyData, playerNumber);
    }
    else
    {
        Debug.LogError("Failed to retrieve the created lobby.");
    }
  }
}