using UnityEngine;
using UnityEngine.UI;
using Unity.Netcode;
using Unity.Services.Authentication;
using Unity.Services.Core;
using Unity.Services.Lobbies;
using Unity.Services.Lobbies.Models;
using System.Threading.Tasks;
using System.Collections.Generic;
using UnityEngine.Events;

// TODO set up the rate limit for calling some of the methods

public class Mult_Lobby : MonoBehaviour
{
  private Lobby hostLobby;
  private Lobby clientLobby;
  private float heartbeatTimer;
  private float lobbyUpdateTimer;
  private float refreshInterval;
  [SerializeField] private Cooldown refreshButton;
  private bool canRefresh = true;
  private string playerName;

  [SerializeField] private Manager_Auth authManager;
  
  // Define an event that other scripts can listen to
  public event UnityAction<Lobby> OnLobbyUpdated;
  private async void Start()
  {
    // Initialize the Unity Services
    await UnityServices.InitializeAsync();

    if (AuthenticationService.Instance.IsSignedIn)
    {
      Debug.Log("Signed in as: " + AuthenticationService.Instance.PlayerId);
    }
    else
    {     
      // Authenticate the player anonymously
      AuthenticationService.Instance.SignedIn += () =>
      {
        Debug.Log("Signed in anonymously " + AuthenticationService.Instance.PlayerId);
      };

      await AuthenticationService.Instance.SignInAnonymouslyAsync();
    }
  }

  private void Update()
  {
    HandleLobbyHeartbeat();
    HandleLobbyPollForUpdate();
  }
  
  // Ping the lobby to avoid being closed due inactivity 
  private async void HandleLobbyHeartbeat()
  {
    if (hostLobby == null)
    {
      return;
    }

    heartbeatTimer -= Time.deltaTime;
    if (heartbeatTimer <= 0)
    {
      float heartBeatTimerMax = 15;
      heartbeatTimer = heartBeatTimerMax;

      await LobbyService.Instance.SendHeartbeatPingAsync(hostLobby.Id);
    }
  }

  // Poll for lobby updates
  private async void HandleLobbyPollForUpdate()
  {
    if (clientLobby == null)
    {
      return;
    }

    lobbyUpdateTimer -= Time.deltaTime;
    if (lobbyUpdateTimer <= 0)
    {
      float lobbyUpdateTimerMax = 1.2f;
      lobbyUpdateTimer = lobbyUpdateTimerMax;

      Lobby lobby = await LobbyService.Instance.GetLobbyAsync(clientLobby.Id);
      clientLobby = lobby;
      
      // Trigger the event when the lobby is updated
      OnLobbyUpdated?.Invoke(clientLobby); // Notify listeners with the updated lobby  
    }
  }

  // Get the lobby Options
  public CreateLobbyOptions GetLobbyOptions(UI_LobbyData.LobbyData lobbyData)
  {
    bool isPrivate = lobbyData.isPrivate;
    string mapName = lobbyData.mapName;
    string mapAuthor = lobbyData.mapAuthor;
    string mapId = lobbyData.mapId;
    string lobbyStatus = "Lobby";

    CreateLobbyOptions lobbyOptions = new CreateLobbyOptions 
    {
      IsPrivate = isPrivate,
      Player = GetPlayer(),
      Data = new Dictionary<string, DataObject>
      {
        { "MapName", new DataObject(DataObject.VisibilityOptions.Public, mapName) },
        { "MapId", new DataObject(DataObject.VisibilityOptions.Public, mapId) },
        { "LobbyStatus", new DataObject(DataObject.VisibilityOptions.Public, lobbyStatus) }
      }
    };

    return lobbyOptions;
  }

  // Create a lobby with its settings
  public async Task CreateLobby(UI_LobbyData.LobbyData lobbyData)
  {
    string lobbyName = lobbyData.lobbyName;
    int maxPlayers = lobbyData.maxPlayers;
    CreateLobbyOptions lobbyOptions = GetLobbyOptions(lobbyData);

    try
    {
      Lobby lobby = await LobbyService.Instance.CreateLobbyAsync(lobbyName, maxPlayers, lobbyOptions);

      // Store the lobby reference
      hostLobby = lobby;
      clientLobby = lobby;

      PrintPlayers(hostLobby);
      Debug.Log("Lobby created: " + lobby.Id + " with name: " + lobby.Name + " and max players: " + lobby.MaxPlayers + " and with lobbyCode: " + lobby.LobbyCode);
    }
    catch (LobbyServiceException e)
    {
      Debug.LogError("Failed to create lobby: " + e.Message);
    }
  }

  public async Task<QueryResponse> ListLobbies()
  {
    if (refreshButton.IsOnCooldown())
    {
      return null;
    }
    // Limit the refresh rate
    refreshButton.StartCooldown(5f);
    try
    {
     /*  // Example of applyting a filter to the query
      QueryLobbiesOptions options = new QueryLobbiesOptions 
      {
        Count = 25,
        Filters = new List <QueryFilter> 
        {
          new QueryFilter 
          (
            QueryFilter.FieldOptions.AvailableSlots,
            "0",
            QueryFilter.OpOptions.GT
          )
        },
        Order = new List <QueryOrder> 
        {
          new QueryOrder 
          (
            false,
            QueryOrder.FieldOptions.Created
          )
        }
      }; */

      QueryResponse queryResponse = await Lobbies.Instance.QueryLobbiesAsync();
      Debug.Log("Lobbies listed: " + queryResponse.Results.Count);
      return queryResponse;
    }
    catch (LobbyServiceException e)
    {
      Debug.LogError("Failed to list lobbies: " + e.Message);
      return null;
    }
  }

  public async Task<Lobby> JoinLobbyByCode(string lobbyCode)
  {
    try
    {
      JoinLobbyByCodeOptions joinLobbyByCodeOptions = new JoinLobbyByCodeOptions 
      {
        Player = GetPlayer()
      };
      // Join the lobby by code
      Lobby lobby = await Lobbies.Instance.JoinLobbyByCodeAsync(lobbyCode, joinLobbyByCodeOptions);
      clientLobby = lobby;

      PrintPlayers(lobby);
      return lobby;
    }
    catch (LobbyServiceException e)
    {
      Debug.LogError("Failed to join lobby: " + e.Message);
      return null;
    }
  }

  public async Task<Lobby> JoinLobbyById(string lobbyCode)
  {
    try
    {
      // Create join options, like passing player data
      JoinLobbyByIdOptions joinLobbyByIdOptions = new JoinLobbyByIdOptions
      {
          Player = GetPlayer() // Ensure this is your player object or data
      };

      // Join the lobby by code
      Lobby lobby = await Lobbies.Instance.JoinLobbyByIdAsync(lobbyCode, joinLobbyByIdOptions);
      clientLobby = lobby;

      PrintPlayers(lobby);
      return lobby;
    }
    catch (LobbyServiceException e)
    {
      Debug.LogError("Failed to join lobby: " + e.Message);
      return null;
    }
  }

  private Player GetPlayer()
  {
    playerName = authManager.GetUserDisplayName();
    Debug.Log("Player name: " + playerName);

    return new Player
    {
      Data = new Dictionary<string, PlayerDataObject>
      {
        { "PlayerName", new PlayerDataObject(PlayerDataObject.VisibilityOptions.Member, playerName) }
      }
    };
  }

  public async void QuickJoinLobby()
  {
    try
    {
      QuickJoinLobbyOptions quickJoinOptions = new QuickJoinLobbyOptions 
      {
        Player = GetPlayer()
      };
      // Quick join a lobby
      Lobby lobby = await LobbyService.Instance.QuickJoinLobbyAsync(quickJoinOptions);
      clientLobby = lobby;
      Debug.Log("Lobby joined: " + lobby.Id + " with name: " + lobby.Name + " and max players: " + lobby.MaxPlayers + " and with lobbyCode: " + lobby.LobbyCode);

      PrintPlayers(lobby);
    }
    catch (LobbyServiceException e)
    {
      Debug.LogError("Failed to quick join lobby: " + e.Message);
    }
  }

  public Lobby GetClientLobby()
  {
    return clientLobby;
  }

  private void PrintPlayers(Lobby lobby)
  {
    Debug.Log("Players in lobby: " + lobby.Name);
    foreach (Player player in lobby.Players)
    {
      Debug.Log("Player found: " + player.Id + " with name: " + player.Data["PlayerName"].Value);
      Debug.Log("Lobby data: " + lobby.Data["MapName"].Value + " with mapId: " + lobby.Data["MapId"].Value);
      Debug.Log("Lobby status: " + lobby.Data["LobbyStatus"].Value);
    }
  }

  public async void UpdateLobbyStatus(string status)
  {
    try
    {
      // Update the lobby status
      hostLobby = await Lobbies.Instance.UpdateLobbyAsync(hostLobby.Id, new UpdateLobbyOptions
      {
        Data = new Dictionary<string, DataObject> 
        {
          { "LobbyStatus", new DataObject(DataObject.VisibilityOptions.Public, status) }
        }
      });

      PrintPlayers(hostLobby);
    }
    catch (LobbyServiceException e)
    {
      Debug.LogError("Failed to quick join lobby: " + e.Message);
    }
  }

  public async void UpdatePlayerName(string name)
  {
    try
    {
      playerName = name;
      await LobbyService.Instance.UpdatePlayerAsync(clientLobby.Id, AuthenticationService.Instance.PlayerId, new UpdatePlayerOptions
      {
        Data = new Dictionary<string, PlayerDataObject>
        {
          { "PlayerName", new PlayerDataObject(PlayerDataObject.VisibilityOptions.Member, name) }
        }
      });
      Debug.Log("Player updated: " + AuthenticationService.Instance.PlayerId + " with name: " + name);
    }
    catch (LobbyServiceException e)
    {
      Debug.LogError("Failed to update player: " + e.Message);
    }
  }

  public async void LeaveLobby()
  {
    try
    {
      await LobbyService.Instance.RemovePlayerAsync(clientLobby.Id, AuthenticationService.Instance.PlayerId);
      Debug.Log("Player left lobby: " + clientLobby.Id);
      clientLobby = null;
    }
    catch (LobbyServiceException e)
    {
      Debug.LogError("Failed to leave lobby: " + e.Message);
    }
  }

  private async void KickPlayer()
  {
    try
    {
      await LobbyService.Instance.RemovePlayerAsync(clientLobby.Id, clientLobby.Players[1].Id);
      Debug.Log("Player left lobby: " + clientLobby.Id);
    }
    catch (LobbyServiceException e)
    {
      Debug.LogError("Failed to leave lobby: " + e.Message);
    }
  }
  
  private async void MigrateLobbyHost()
  {
    try
    {
      hostLobby = await Lobbies.Instance.UpdateLobbyAsync(hostLobby.Id, new UpdateLobbyOptions
      {
        HostId = clientLobby.Players[1].Id
      });
      clientLobby = hostLobby;

      PrintPlayers(hostLobby);
      LeaveLobby();
      Debug.Log("Player left lobby: " + clientLobby.Id);
    }
    catch (LobbyServiceException e)
    {
      Debug.LogError("Failed to migrate host: " + e.Message);
    }
  }

  private async void DeleteLobby()
  {
    try
    {
      await LobbyService.Instance.DeleteLobbyAsync(clientLobby.Id);
      Debug.Log("Lobby deleted: " + clientLobby.Id);
    }
    catch (LobbyServiceException e)
    {
      Debug.LogError("Failed to delete lobby: " + e.Message);
    }
  }


}