## License

This project is released under a custom academic-use license. See the [LICENSE](LICENSE) file for full terms.

# Branching Strategy
We use a simple and efficient branching strategy to manage our development workflow. This approach ensures clean code integration, easy testing, and smooth releases.

## Branches
1. `main`:

    - Represents production-ready code.

    - Only updated via merges from development.

    - Tagged with version numbers (e.g., v1.0.0) for releases.

2. `development`:

    - Primary branch for ongoing work.

    - All user stories (US) are developed and tested here.

    - Acts as the integration branch for smaller tasks or single-part user stories.

3. Feature Branches (e.g., US_x): 

   - Created only when a user story is too large or needs to be split into parallel work.

   - Work is done in the feature branch, then merged back into development for integration and testing.

# Decisions
| Title                            | Problem Description                                                    | Decision Made                               | Description                                                                                                                                                                                                      |
| -------------------------------- | ---------------------------------------------------------------------- | ------------------------------------------- | ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- |
| 0.1 Firebase Integration         | Need reliable backend for user data, analytics, and real-time features | Use Firebase                                | Firebase provides user authentication, database (Firestore), cloud functions, and analytics out-of-the-box. It fits well with Unity, scales easily, and works on the free tier for small to mid-scale games.     |
| 0.2 Subfirebase Compression      | Data size impacts upload/download speed and storage costs              | Compress string data before upload          | Implemented a custom GZip-based method `CompressString(string text)` to reduce data size before uploading to Firebase. This improves speed and minimizes quota usage, especially beneficial under the free tier. |
| 0.3 Easy Save 3                  | Manual serialization is time-consuming and error-prone                 | Use Easy Save 3 for local save/load         | Easy Save 3 enables encrypted local data persistence with minimal code. It simplifies managing local game state, progress, and preferences, improving reliability and reducing development time.                 |
| 0.4 Joystick Input               | Native input systems are limited for touch and mobile devices          | Use Joystick Pack plugin                    | Added Joystick Pack for Unity to improve mobile/touch controls. It provides customizable on-screen joysticks and works well with Unity's Input System.                                                           |
| 0.5 Multiplayer Support          | Need scalable multiplayer with Unity compatibility                     | Use Unity's NGO (Netcode for GameObjects)   | NGO is Unity’s official multiplayer framework, offering tight integration with the Unity engine and flexibility for custom transport layers. Chosen for long-term maintainability and community support.         |
| 0.6 Unity IAP                    | Monetization requires in-app purchases                                 | Integrate Unity IAP                         | Unity IAP allows implementing cross-platform in-app purchases with a unified API. It supports Google Play and App Store, handling store-specific logic and validation.                                           |
| 0.7 Offline Mode                 | Game must support some features offline                                | Cache essential data locally with Easy Save | User progress and non-dynamic data are saved locally using Easy Save 3. On startup, the game checks for network connectivity. If the player is logged in and an account exists, currency and other syncable data are updated from Firebase. Otherwise, the local cache is used. This ensures smooth offline gameplay with automatic synchronization when back online.                                           |
| 0.8 Map Upload/Download Bandwidth Optimization | Mobile networks often have limited bandwidth, and large map data can cause slow or failed uploads/downloads | Upload maps as single documents to Firebase Storage and batch load on download | Map data is compressed and uploaded as a single document to **Firebase Storage**, not Firestore. In Firestore, only a reference (URL) to the stored file is saved. This keeps Firestore lean and optimized for queries. On the client side, **map downloads are batched** when the user accesses them, improving performance and reducing redundant network calls. |




# Unity C# Project Structure and naming
The `4_Scripts` folder contains the core scripts for the Golfever project, categorized into 5 main sections, each serving a distinct feature set.

## 1. 0_Utilities
This folder contains utility scripts that provide reusable components for data processing, error handling, extensions, key management, and transitions.

- **DataProcessing:**
  - `CompressionUtility.cs` - Handles data compression.
  - `EncodingUtility.cs` - Handles data encoding.
  
- **ErrorHandling:**
  - `DataIntegrityChecker.cs` - Ensures the integrity of data.
  
- **Extensions:**
  - `EnumExtension.cs` - Extends functionality for enumerations.

- **KeysHandling:**
  - `FirestoreKeys.cs` - Manages Firestore-specific keys.
  - `LocalSaveKeys.cs` - Manages local save keys.
  - `Manager_Key.cs` - Manages overall key handling logic.

- **Transitions:**
  - `Cooldown.cs` - Handles cooldown logic.
  - `Manager_SceneTransitor.cs` - Manages scene transitions.
  - `UI_Cooldown.cs` - UI component for displaying cooldown.


## 2. 1_Player
This directory holds player-related scripts including controllers, interactions, and skins, with an additional folder for player enhancers.

- **Enhancers:**
  - `Booster.cs` - Base class for boosters.
  
  - **Boosters:**
    - `Booster_Comet.cs` - A specific booster for the player.
  
  - `Skill.cs` - Base class for player skills.
  
  - **Skills:**
    - `Skill_None.cs` - A default skill class.
    
- **PlayerController.cs** - Controls player movement and actions.
- **PlayerInteraction.cs** - Handles player interactions with the environment.
- **PlayerSkin.cs** - Manages player skin customization.
- **UI_PlayerInteraction.cs** - UI elements related to player interactions.


## 3. 2_Map
This section contains scripts related to map data, levels, and interactions. It is further divided into two main parts: `Data` and `Level`.

- **Data:**
  - **Display:**
    - `Object_MapDisplayCloud.cs` - Handles cloud-based map display.
    - `Object_MapDisplay.cs` - Handles local map display.
    - `UI_MapDataDisplayCloud.cs` - UI for cloud-based map data display.
    - `UI_MapDataDisplay.cs` - UI for local map data display.
    
  - **LocalPersistence:**
    - `CRUD_EasySave3.cs` - Handles CRUD operations using EasySave.
    - `Manager_IndexData.cs` - Manages index data.
    - `Manager_TilesData.cs` - Manages tile data.
    - `Processing_TilesData.cs` - Processes tile data.
    
  - **Model:**
    - `Interact_MapDataDisplayCloud.cs` - Handles cloud map data interactions.
    - `Interact_MapDataDisplayLocal.cs` - Handles local map data interactions.
    - `Interact_MapDisplayBase.cs` - Base class for map data interactions.
    - `Processing_MapData.cs` - Processes map data.
    
  - `Manager_MapDataCloud.cs` - Manages cloud-based map data.
  - `Manager_MapDataLocal.cs` - Manages local map data.
  - `Manager_MapMenus.cs` - Manages map-related menus.
  - `Struct_MapData.cs` - Structure definition for map data.
  
- **Level:**
  - **Building:**
    - `BuildingButtonHandler.cs` - Manages building button interactions.
    - `BuildingCreator.cs` - Creates buildings.
    - `BuildingHistory.cs` - Stores building history.
    - `BuildingHistoryHandler.cs` - Manages building history.
    - `BuildingInputs.cs` - Handles building-related inputs.
    - `BuildingObjectBase.cs` - Base class for building objects.
  
  - **GameSetup:**
    - `GameSetters.cs` - Handles game setup logic.
    - `LevelSelector.cs` - Manages level selection.
    - `LevelSetter.cs` - Sets up the selected level.
    - `Manager_LevelEditor.cs` - Manages the level editor.
    
  - **Interaction:**
    - `Manager_Camera.cs` - Controls camera interactions.
    - `PanZoom.cs` - Handles panning and zooming.
    - `RotationController.cs` - Handles rotation controls.
    
  - **ObjectsSetup:**
    - `ObjectProperties.cs` - Manages properties of in-game objects.


## 4. 3_Cloud
This folder holds scripts related to cloud services, such as Firebase authentication and storage.

- **Auth:**
  - `CRUD_Authentication.cs` - Manages CRUD operations for user authentication.
  - `UI_Authentication.cs` - UI for user authentication.
  
- **FirebaseInitializer.cs** - Initializes Firebase services.
- **Manager_Auth.cs** - Manages authentication processes.
- **Manager_MapAccess.cs** - Manages cloud-based map access.
- **Manager_MapQuery.cs** - Manages cloud-based map queries.
  
- **Storage:**
  - `CloudMapFetch.cs` - Fetches maps from the cloud.
  - `CRUD_Firestore.cs` - Manages CRUD operations in Firestore.



## 5. 4_Multiplayer
Currently, this section is not populated with any scripts but is likely reserved for future development of multiplayer features.

---

## Interaction Flow Overview

In this project, the core interaction between the files and scripts revolves around **managers**. Managers serve as the central units responsible for handling data, controlling UI-related logic, and managing data processing and error handling. Below is a structured breakdown of these interactions:

### 1. Managers as Central Controllers
- **Managers** are the core components responsible for:
  - **Handling Data:** They receive inputs from various sources, primarily **UI files**.
  - **Managing UI Logic:** They control all user interface-related logic, such as menus and inputs.
  - **Data Processing:** Managers ensure that data is processed correctly by referencing and calling **processing scripts**.

### 2. UI Interaction
- The **UI files** are responsible for controlling the interface, including user interactions, input handling, and menu operations.
- These files pass relevant data to their associated manager.
- The manager is responsible for further processing the UI inputs and directing them to the appropriate **processing scripts**.

### 3. Processing Scripts
- The **processing scripts** handle the logic that processes and transforms data.
- Managers reference these scripts as needed, allowing them to handle complex operations without overloading the UI logic.
- **Processing scripts** are allowed to **call other processing scripts**, when it makes sense, to maintain modularity and reusability of the code.

### 4. Error Handling
- Each **manager** is responsible for its own **error handling**.
- If errors occur during data processing or UI interactions, the manager ensures that the errors are caught, processed, and resolved by referencing the appropriate **error-handling scripts**.

### 5. Manager-to-Manager Communication
- Managers can communicate directly with other managers to pass or retrieve data, allowing for centralized control and minimal redundancy.
- While managers can reference and call **UI** and **processing scripts** directly, it is **recommended to pass data between managers** wherever possible. This ensures centralized data flow and minimizes inconsistencies.


## Visual Representation

Here's the diagram representing the interaction flow:

![Example of file interaction](./ReadmeImages/fileInteraction.png)

---

# Naming Convention Guide

This project follows specific naming conventions to ensure consistency, readability, and maintainability across all scripts and files. The conventions are based on Unity’s best practices for naming components and methods, adapted to the project’s structure.

## 1. **General Naming Guidelines**
- **PascalCase** for class names, public methods, and public variables.
- **camelCase** for private or serialized fields.
- Use **descriptive** and **concise** names that indicate the functionality.
- Include a relevant **prefix** or **suffix** where necessary, based on the role of the component (e.g., `Manager_`, `UI_`, `Interact_`).

## 2. **Manager Classes**
- Manager classes are responsible for controlling or handling high-level processes and interactions.
- Use the `Manager_` prefix for all manager classes.

### Examples:
- `Manager_MapDataLocal`
- `Manager_SceneTransitor`
- `Manager_IndexData`

### Guidelines:
- The name should indicate the specific area it manages.
- If the manager handles multiple features or systems, append the main focus area of the class after `Manager_`.

## 3. **UI-Related Files**
- UI files handle the user interface logic, including input, menus, and interactive components.
- Use the `UI_` prefix for classes that directly interact with UI elements.

### Examples:
- `UI_MapDataDisplay`
- `UI_Cooldown`
- `UI_Authentication`

### Guidelines:
- Use descriptive names that include the type of UI interaction they control.
- This helps distinguish UI logic from core data and processing scripts.

## 4. **Processing Scripts**
- Processing scripts handle the transformation, validation, or manipulation of data.
- Use descriptive names that clearly state what kind of processing is being performed.
- It is acceptable for **processing scripts** to call **other processing scripts** if it makes sense for modularity and reusability.

### Examples:
- `MapDataProcessing`
- `TilesDataSave`
- `Processing_MapData`

### Guidelines:
- Use clear, action-oriented names such as `Process`, `Save`, `Load`, `Update`, or `Handle`.
- Avoid overly generic terms like `DataHandler` or `ProcessScript`.

## 5. **Interaction Classes**
- Interaction classes represent specific interactions with data, often involving data flow between different systems.
- Use the `Interact_` prefix for scripts that handle interaction between data systems.

### Examples:
- `Interact_MapDataDisplayLocal`
- `Interact_MapDataDisplayCloud`

### Guidelines:
- Clearly define the source and destination of interactions in the class name.
- Avoid abbreviations unless they are commonly understood (e.g., `Map`, `UI`).

## 6. **Error Handling**
- Error-handling scripts manage the detection and resolution of errors.
- Name error-handling scripts descriptively to indicate what aspect they validate or check.

### Examples:
- `DataIntegrityChecker`
- `ErrorLogger`

### Guidelines:
- Append the purpose of error checking in the name, such as `Validator`, `Checker`, or `Handler`.

## 7. **Serialization and Data Keys**
- For classes or fields that handle serialization, data keys, or storage-related tasks, use relevant naming that indicates the type of storage or key management involved.
  
### Examples:
- `FirestoreKeys`
- `LocalSaveKeys`
- `KeyManager`

### Guidelines:
- Use the `Keys` suffix for classes handling key storage or retrieval.
- Use the `Save` or `Storage` suffix for classes that deal with saving and persistence of data.

## 8. **CRUD Methods**
CRUD (Create, Read, Update, Delete) methods should have descriptive names indicating their purpose and operation. Here's how they should be named:

### Examples:
- **Create:** `CreateMap`, `DuplicateMap`
- **Read:** `TryGetMapDetails`
- **Update:** `UpdateMapData`, `UpdateUploadStatus`
- **Delete:** `DeleteMap`

### Guidelines:
- Use the action in the method name (e.g., `Create`, `Read`, `Update`, `Delete`) followed by a descriptive noun.
- Avoid overloading method names unless necessary.

## 9. **Scene and Transition Classes**
- Classes responsible for scene management or transitions between scenes should use descriptive names with the `Manager_` prefix or `Transitor_` suffix.

### Examples:
- `Manager_SceneTransitor`
- `SceneTransitorManager`

### Guidelines:
- Make sure the name clearly reflects its responsibility for transitioning or loading scenes.
---
