import uuid
from typing import Any

from fastapi import APIRouter, Depends, HTTPException, UploadFile
from sqlmodel import col, delete, func, select, or_

from app import crud
from app.api.deps import (
    CurrentUser,
    SessionDep,
    get_current_active_superuser,
    get_current_teacher
)
from app.core.config import settings
from app.core.security import get_password_hash, verify_password
from app.models import (
    Message,
    UserUpdatePassword,
    User,
    UserCreate,
    UserPublic,
    UserRegister,
    UsersPublic,
    UserUpdate,
    UserUpdateMe,
    Course
)
from app.utils import generate_new_account_email, send_email

UPLOAD_DIR = './api/corrections'

router = APIRouter()


@router.get(
    "/",
    dependencies=[Depends(get_current_active_superuser)],
    response_model=UsersPublic,
)
def read_users(session: SessionDep, skip: int = 0, limit: int = 100) -> Any:
    """
    Retrieve users.
    """

    count_statement = select(func.count()).select_from(User)
    count = session.exec(count_statement).one()

    statement = select(User).offset(skip).limit(limit)
    users = session.exec(statement).all()

    return UsersPublic(data=users, count=count)

@router.get(
    "/students",
    dependencies=[Depends(get_current_teacher)],
    response_model=UsersPublic,
)
def read_students_users(
    session: SessionDep, 
    skip: int = 0, 
    limit: int = 100,
    search: str = None
) -> Any:
    """
    Retrieve only student users with optional search functionality.
    """
    base_query = select(User).where(User.is_student == True)
    
    if search:
        search_term = f"%{search}%"
        base_query = base_query.where(
            (User.email.ilike(search_term)) |
            (User.name.ilike(search_term)) |
            (User.surnames.ilike(search_term)) |
            (User.niub.ilike(search_term))
        )
    
    count_query = select(func.count()).select_from(
        base_query.subquery()
    )
    count = session.exec(count_query).one()
    
    students_query = base_query.offset(skip).limit(limit)
    students = session.exec(students_query).all()
    
    return UsersPublic(data=students, count=count)

@router.post(
    "/", dependencies=[Depends(get_current_active_superuser)], response_model=UserPublic
)
def create_user(*, session: SessionDep, user_in: UserCreate) -> Any:
    """
    Create new user.
    """
    user = crud.user.get_user_by_email(session=session, email=user_in.email)
    if user:
        raise HTTPException(
            status_code=400,
            detail="The user with this email already exists in the system.",
        )

    user = crud.user.create_user(session=session, user_create=user_in)
    if settings.emails_enabled and user_in.email:
        email_data = generate_new_account_email(
            email_to=user_in.email, username=user_in.email, password=user_in.password
        )
        send_email(
            email_to=user_in.email,
            subject=email_data.subject,
            html_content=email_data.html_content,
        )
    return user


@router.patch("/me", response_model=UserPublic)
def update_user_me(
    *, session: SessionDep, user_in: UserUpdateMe, current_user: CurrentUser
) -> Any:
    """
    Update own user.
    """

    if user_in.email:
        existing_user = crud.user.get_user_by_email(session=session, email=user_in.email)
        if existing_user and existing_user.niub != current_user.niub:
            raise HTTPException(
                status_code=409, detail="USER_EMAIL_EXISTS"
            )
    user_data = user_in.model_dump(exclude_unset=True)
    current_user.sqlmodel_update(user_data)
    session.add(current_user)
    session.commit()
    session.refresh(current_user)
    return current_user


@router.patch("/me/password", response_model=Message)
def update_password_me(
    *, session: SessionDep, body: UserUpdatePassword, current_user: CurrentUser
) -> Any:
    """
    Update own password.
    """
    if not verify_password(body.current_password, current_user.hashed_password):
        raise HTTPException(status_code=400, detail="Incorrect password")
    if body.current_password == body.new_password:
        raise HTTPException(
            status_code=400, detail="NEW_PASSWORD_SAME_AS_CURRENT_ONE"
        )
    hashed_password = get_password_hash(body.new_password)
    current_user.hashed_password = hashed_password
    session.add(current_user)
    session.commit()
    return Message(message="Password updated successfully")


@router.get("/me", response_model=UserPublic)
def read_user_me(current_user: CurrentUser) -> Any:
    """
    Get current user.
    """
    return current_user


@router.delete("/me", response_model=Message)
def delete_user_me(session: SessionDep, current_user: CurrentUser) -> Any:
    """
    Delete own user.
    """
    if current_user.is_admin:
        raise HTTPException(
            status_code=403, detail="Super users are not allowed to delete themselves"
        )
    session.delete(current_user)
    session.commit()
    return Message(message="User deleted successfully")


@router.post("/signup", response_model=UserPublic)
def register_user(session: SessionDep, user_in: UserRegister) -> Any:
    """
    Create new user without the need to be logged in.
    """
    user_email = crud.user.get_user_by_email(session=session, email=user_in.email)
    user_niub = crud.user.get_user_by_niub(session=session, niub=user_in.niub)
    if user_email and user_niub:
        raise HTTPException(
            status_code=400,
            detail="EMAIL_AND_NIUB_ALREADY_EXISTS",
        )
    elif user_email:
        raise HTTPException(
            status_code=400,
            detail="EMAIL_ALREADY_EXISTS",
        )
    elif user_niub:
        raise HTTPException(
            status_code=400,
            detail="NIUB_ALREADY_EXISTS",
        )
    user_create = UserCreate.model_validate(user_in)
    user = crud.user.create_user(session=session, user_create=user_create)
    
    stmt = select(Course).where(
        or_(
            Course.pending_niubs == user.niub,
            Course.pending_niubs.like(f"{user.niub},%"),
            Course.pending_niubs.like(f"%,{user.niub},%"),
            Course.pending_niubs.like(f"%,{user.niub}")
        )
    )
    pending_courses = session.exec(stmt).all()

    for course in pending_courses:
        course.users.append(user)
        for practice in course.practices:
            if user not in practice.users:
                practice.users.append(user)
                session.add(practice)

        pending_niubs_list = [niub for niub in course.pending_niubs.split(",") if niub != user.niub]
        course.pending_niubs = ",".join(pending_niubs_list)

        session.add(course)

    session.commit()

    return user


@router.get("/{user_niub}", response_model=UserPublic)
def read_user_by_id(
    user_niub: str, session: SessionDep, current_user: CurrentUser
) -> Any:
    """
    Get a specific user by niub.
    """
    user = session.get(User, user_niub)
    if user == current_user:
        return user
    if not current_user.is_admin:
        raise HTTPException(
            status_code=403,
            detail="The user doesn't have enough privileges",
        )
    return user


@router.patch(
    "/{user_niub}",
    dependencies=[Depends(get_current_active_superuser)],
    response_model=UserPublic,
)
def update_user(
    *,
    session: SessionDep,
    user_niub: str,
    user_in: UserUpdate,
) -> Any:
    """
    Update a user.
    """

    db_user = session.get(User, user_niub)
    if not db_user:
        raise HTTPException(
            status_code=404,
            detail="The user with this id does not exist in the system",
        )
    if user_in.email:
        existing_user = crud.user.get_user_by_email(session=session, email=user_in.email)
        if existing_user and existing_user.niub != user_niub:
            raise HTTPException(
                status_code=409, detail="User with this email already exists"
            )

    db_user = crud.user.update_user(session=session, db_user=db_user, user_in=user_in)
    return db_user


@router.delete("/{user_niub}", dependencies=[Depends(get_current_active_superuser)])
def delete_user(
    session: SessionDep, current_user: CurrentUser, user_niub: str
) -> Message:
    """
    Delete a user.
    """
    user = session.get(User, user_niub)
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    if user == current_user:
        raise HTTPException(
            status_code=403, detail="Super users are not allowed to delete themselves"
        )
    session.delete(user)
    session.commit()
    return Message(message="User deleted successfully")